--- GPIO Test Core
---
--- Top Level Test Code
---
--- conor.orourke@scorpionnetworks.com
--- 15 October 2012
---


-- Library Declarations

library WORK;
library IEEE;
library UNISIM;

use IEEE.std_logic_1164.all; -- basic IEEE library
use IEEE.numeric_std.all;    -- IEEE library for the unsigned type and arith

use UNISIM.VCOMPONENTS.ALL;

use WORK.MityDSP_L138_pkg.ALL;


-- Entity

entity Gpio_Test_Top is

    generic (
        BOARD_REV     : string := "C";
        DECODE_BITS   : integer range 1 to 9 := 5
    );

    port (
        i_ema_clk     : in std_logic; -- 100 MHz EMIF clock

        -- DSP EMIFA BUS Interface
        i_ema_cs0_n   : in std_logic; -- RESERVED for SDRAM Controller
        i_ema_cs2_n   : in std_logic; -- ARM core CE space
        i_ema_cs3_n   : in std_logic; -- NAND FLASH space, not used
        i_ema_cs4_n   : in std_logic; -- DSP core CE space
        i_ema_cs5_n   : in std_logic; -- Reserved core space, not used
        i_ema_oe_n    : in std_logic;
        i_ema_we_n    : in std_logic;
        io_ema_wait_n : inout std_logic_vector(1 downto 0);
        io_ema_d      : inout std_logic_vector(15 downto 0);
        i_ema_a       : in std_logic_vector(13 downto 0);
        i_ema_ba      : in std_logic_vector(1 downto 0);
        i_ema_wen_dqm : in std_logic_vector(1 downto 0); 
        i_ema_cas     : in std_logic; -- reserved for SDRAM controller, not used
        i_ema_ras     : in std_logic; -- reserved for SDRAM controller, not used
        i_ema_sdcke   : in std_logic; -- reserved for SDRAM controller, not used
        i_ema_rnw     : in std_logic; -- reserved for SDRAM controller, not used

        -- DSP IRQ lines
        o_int         : out std_logic_vector(1 downto 0);
        o_nmi_n       : out std_logic;

        -- GPIO Test Lines
        io_j702_3_gpio : inout std_logic;
        io_j702_4_gpio : inout std_logic
    );

end Gpio_Test_Top;


-- Architecture

architecture gpio_arch of Gpio_Test_Top is

    -- Constants --

    constant FPGA_APPLICATION_ID  : std_logic_vector(7 downto 0) := "00000000";
    constant FPGA_VERSION_MAJOR   : std_logic_vector(3 downto 0) := "0001";
    constant FPGA_VERSION_MINOR   : std_logic_vector(3 downto 0) := "0000";
    constant FPGA_YEAR            : std_logic_vector(4 downto 0) := "01010";
    constant FPGA_MONTH           : std_logic_vector(3 downto 0) := "0010";
    constant FPGA_DAY             : std_logic_vector(4 downto 0) := "10100";

    -- CPU Mapping: ARM is INT0
    --              DSP is INT1

    constant CORE_BASE_MODULE     : integer := 0;
    constant CORE_GPIO_MODULE     : integer := 5;

    constant CORE_GPIO_IRQ_LEVEL  : integer := 0;
    constant CORE_GPIO_IRQ_VECTOR : integer := 0;

    -- Clock Signals --

    signal Ema_clk    : std_logic;

    -- EMIFA Signals

    signal Ema_d      : std_logic_vector(15 downto 0);
    signal Ema_wait_n : std_logic_vector(1 downto 0) := "11";
    signal T_ema_wait : std_logic;
    signal T_ema_d    : std_logic;

    signal Irq_map    : bus16_vector(1 downto 0) := (others=>(others=>'0')); 

    -- Core Logic signals

    signal Be_r       : std_logic_vector(1 downto 0);
    signal Addr_r     : std_logic_vector(5 downto 0);
    signal Arm_cs5_r  : std_logic_vector((2**DECODE_BITS)-1 downto 0);
    signal Dsp_cs4_r  : std_logic_vector((2**DECODE_BITS)-1 downto 0);
    signal Edi_r      : std_logic_vector(15 downto 0);
    signal Edo_arm    : bus16_vector((2**DECODE_BITS)-1 downto 0);
    signal Edo_dsp    : bus16_vector((2**DECODE_BITS)-1 downto 0);
    signal Rd_r       : std_logic;
    signal Wr_r       : std_logic;


begin

    ---------------
    -- PORT MAPS --
    ---------------

    -- Route i_ema_clk via Global Net (Xilinx specific) to EMA_CLK

    emaclk_inst : BUFG
        port map (O => Ema_clk,
                  I => i_ema_clk);


    -----------------
    -- EMIF_Iface --
    ----------------

    emifa_iface_inst : EMIFA_iface
        generic map ( 
            DECODE_BITS       => DECODE_BITS, 
            CONFIG            => "MityDSP_L138"
        )
        port map (
            ema_clk   => Ema_clk,
            i_ema_cs0_n   => i_ema_cs0_n,
            i_ema_cs2_n   => i_ema_cs2_n,
            i_ema_cs3_n   => i_ema_cs3_n,
            i_ema_cs4_n   => i_ema_cs4_n,
            i_ema_cs5_n   => i_ema_cs5_n,
            i_ema_oe_n    => i_ema_oe_n,
            i_ema_we_n    => i_ema_we_n,
            o_ema_wait_n  => Ema_wait_n,

            t_ema_wait    => T_ema_wait,
            i_ema_d       => io_ema_d, 
            o_ema_d       => Ema_d, 
            t_ema_d       => T_ema_d,

            i_ema_a       => i_ema_a, 
            i_ema_ba      => i_ema_ba, 
            i_ema_wen_dqm => i_ema_wen_dqm,  
            i_ema_cas     => i_ema_cas,  
            i_ema_ras     => i_ema_ras,  
            i_ema_sdcke   => i_ema_sdcke,  
            i_ema_rnw     => i_ema_rnw,  

            -- Internal Core Bus Signals

            o_core_be      => Be_r,  
            o_core_addr    => Addr_r,  
            o_core_cs5     => Arm_cs5_r,  
            o_core_cs4     => Dsp_cs4_r,  
            o_core_edi     => Edi_r,  
            i_core_edo5    => Edo_arm,  
            i_core_edo4    => Edo_dsp,  
            o_core_rd      => Rd_r,  
            o_core_wr      => Wr_r
        );
    -- end emififace_inst

    -- Port Assignments for EMIFA:

    o_nmi_n <= '1';

    io_ema_d <= Ema_d when T_ema_d = '0' else (others=>'Z');

    io_ema_wait_n <= Ema_wait_n when T_ema_wait = '0' else (others=>'Z');


    -----------------
    -- Base Module --
    -----------------

    base_core_inst : base_module
        generic map (
            CONFIG => "MityDSP_L138"
        )
        port map (
            ema_clk         => Ema_clk,
            i_cs            => Arm_cs5_r(CORE_BASE_MODULE),
            i_ID            => FPGA_APPLICATION_ID,
            i_version_major => FPGA_VERSION_MAJOR,
            i_version_minor => FPGA_VERSION_MINOR,
            i_year          => FPGA_YEAR,
            i_month         => FPGA_MONTH,
            i_day           => FPGA_DAY,
            i_ABus          => Addr_r,
            i_DBus          => Edi_r,
            o_DBus          => Edo_arm(CORE_BASE_MODULE),
            i_wr_en         => Wr_r,
            i_rd_en         => Rd_r,
            i_be_r          => Be_r,

            o_dcm_reset     => open,
            i_dcm_status    => "000",
            i_dcm_lock      => '0',

            i_irq_map       => Irq_map,
            o_irq_output    => o_int
        );
    -- end base_core_inst


    -----------------
    -- GPIO Module --
    -----------------

    gpio_core_inst : gpio
        generic map (
            NUM_BANKS       => 1,
            NUM_IO_PER_BANK => 2
        )
        port map (
            clk             => Ema_clk,
            i_ABus          => Addr_r,
            i_DBus          => Edi_r,
            o_DBus          => Edo_arm(CORE_GPIO_MODULE),
            i_wr_en         => Wr_r,
            i_rd_en         => Rd_r,
            i_cs            => Arm_cs5_r(CORE_GPIO_MODULE),
            o_irq           => Irq_map(CORE_GPIO_IRQ_LEVEL)(CORE_GPIO_IRQ_VECTOR),
            i_ilevel        => std_logic_vector(to_unsigned(CORE_GPIO_IRQ_LEVEL, 2)),
            i_ivector       => std_logic_vector(to_unsigned(CORE_GPIO_IRQ_VECTOR, 4)),

            -- Output only, not tristated. Input copy of output
            i_io(0)         => io_j702_3_gpio,
            i_io(1)         => io_j702_4_gpio,
            t_io            => open,
            o_io(0)         => io_j702_3_gpio,
            o_io(1)         => io_j702_4_gpio,

            i_initdir       => "11",
            i_initoutval    => "11"
        );
    -- end gpio_core_inst

end gpio_arch;


