----------------------------------------------------------------------------------
-- Company: ASTM/OmegaTi
-- Engineer:Michele Canepa
-- Module Name:acq_eng_seq_top.vhd 
-- Arch: Behavioral

-- Acquisition Engine, top module, suitable to interface PulSAR family ADC's from
-- Analog Devices. Includes a Block RAM to store the data coming from the ADC, the
-- ADC interface, and the Address Generator. 

-- Done signal from ADC interface and write enable has a clock cycle latency, in order
-- to let the address increment after writing on the RAM.
----------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

-- Uncomment the following library declaration if instantiating
-- any Xilinx primitives in this code.
library UNISIM;
use UNISIM.VComponents.all;

entity acq_eng_seq_top is
port(
-- SPI ADC Interface Signals
o_sclk: out std_logic;
i_miso: in std_logic; 
i_busy: in std_logic;
-- Emif Clock
ema_clk: in std_logic;
-- To CPU Interrupt Lines: end of frame signals
o_eofa : out std_logic;
o_eofb : out std_logic;
--From Acq Controller
i_framedepth: in std_logic_vector(2 downto 0);
i_rst: in std_logic;
i_cs: in std_logic;
-- Memory Interface Signals
i_addrb : in std_logic_vector(9 downto 0);
i_conv: in std_logic;
o_datab : out std_logic_vector(15 downto 0);
done: out std_logic;
shift_en: out std_logic-- Debug signal
);

end acq_eng_seq_top;

architecture Behavioral of acq_eng_seq_top is

component spi_core_seq is
    Port ( CLOCK : in  STD_LOGIC;
           RESET : in  STD_LOGIC;
           MISO : in  STD_LOGIC;
           SCLK : out  STD_LOGIC;
			  PARALLEL_OUT : out STD_LOGIC_VECTOR(15 downto 0);
			  DONE: out std_logic;
			  SHIFT_EN: out std_logic;-- Debug signal
			  CONV_INPUT: in std_logic);			  
end component;

component spi_addrgen is
    port ( clk: in std_logic;
	        done : in  std_logic;
			  rst: in std_logic;
           eofa : out  std_logic;
           eofb : out  std_logic;
           addr : out  std_logic_vector (9 downto 0);
			  frame_depth: in std_logic_vector(2 downto 0)
			  );
end component;

component memory
  port (
    clka : in std_logic;
    wea : in std_logic_vector(0 downto 0);
    addra : in std_logic_vector(9 downto 0);
    dina : in std_logic_vector(15 downto 0);
    clkb : in std_logic;
    addrb : in std_logic_vector(9 downto 0);
    doutb : out std_logic_vector(15 downto 0)
  );
end component;

signal t_done: std_logic_vector(0 downto 0):=(others=>'0');
signal t_wea: std_logic_vector(0 downto 0):=(others=>'0');
signal t_datarx:std_logic_vector(15 downto 0);
signal t_addra:std_logic_vector(9 downto 0);
signal t_dataout:std_logic_vector(15 downto 0);
signal eofa_r: std_logic:='0'; -- Per introdurre ritardo su end of frame
signal eofb_r: std_logic:='0';

begin

spi_core_inst: spi_core_seq port map(
	CLOCK => ema_clk,
	RESET=> i_rst,
	MISO => i_miso,
	SCLK => o_sclk,
	PARALLEL_OUT=>t_datarx,
	DONE=>t_done(0),
	SHIFT_EN=>shift_en, -- Debug signal
	CONV_INPUT=>i_conv
);
	
spi_addrgen_inst: spi_addrgen PORT MAP(
      clk => ema_clk,
		done => t_wea(0),	-- One-cycle delayed after t_done.
		rst => i_rst,
		eofa => eofa_r,
		eofb => eofb_r,
		addr => t_addra,
		frame_depth => i_framedepth 
	);
	
MEMORY_inst : MEMORY
  PORT MAP (
    clka => ema_clk,
    wea => t_done,
    addra => t_addra,
    dina => t_datarx,
    clkb => ema_clk,
    addrb => i_addrb,
    doutb => t_dataout
  );

-- A flip-flop to delay t_wea signal in respect to t_done signal.

wea_dly: process(ema_clk,t_done)
begin
if rising_edge(ema_clk) then
t_wea<=t_done;
end if;
end process;

-- End of frame flip-flop to delay.

eof_dly: process(ema_clk, eofa_r, eofb_r)
begin
if rising_edge(ema_clk) then
o_eofa<=eofa_r;
o_eofb<=eofb_r;
end if;
end process;

-- This process synchronizes readings and implements chip select. 

reg_read: process(ema_clk) 
begin
if rising_edge(ema_clk) then
	if i_cs = '1' then
	o_datab <= t_dataout;
	else
   o_datab <= (others=>'0');
	end if;
end if;	
end process;	

-- Done is a single signal, t_done is a vector(0 downto 0) 
	
done<=t_done(0);

end Behavioral;

