/**
 * \file spidevice.h
 *
 * \brief Wrapper class to communicate to device via spidev
 *
 * Websites:
 * Critical Link - http://www.criticallink.com
 *
 * \copyright Critical Link LLC 2015
 */

#ifndef SPIDEVICE_H_
#define SPIDEVICE_H_

#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/spi/spidev.h>

#include <string>

struct tsSPIConfiguration
{
	int mnMode;	// SPI transfer mode including polarity and clock phase.
	unsigned char mnLsbType;      // LSB first/last.
	unsigned char mnBits;			// The number of bits per word.
	long mnSpeed;		// The speed in Hz.
};

class tcSPIDevice
{
public:
	/**
	 * Creates a spi device on the given bus and chip select.
	 */
	tcSPIDevice(std::string asDevice);
	tcSPIDevice(const char* asDevice);
	virtual ~tcSPIDevice();

	/**
	 * Open the device if it is not already open.
	 */
	void openDevice();

	/**
	 * Close the SPI device.
	 */
	void closeDevice();

	bool isOpen();

	tsSPIConfiguration getConfiguration();

	bool setConfiguration(struct tsSPIConfiguration &arConfiguration);

	/**
	 * Write the data out.
	 *
	 * @param apTxData - the data to send.
	 * @param anLen - the number of bytes in apTxData
	 */
	void write(char *apTxData, long anLen);

	/**
	 * Read data.
	 *
	 * @param apRxData - the location to store data to
	 * @param anLen - the number of bytes in apRxData
	 */
	void read(char *apRxData, long anLen);

	/**
	 * Perform a full duplex operation; Tx/Rx of length anLen.
	 *
	 * Note: The arrays must be of the same length.
	 *
	 * @param apRxData - the location to store received data in.
	 * @param apTxData - the data to send
	 * @param anLen the length of the arrays.
	 */
	void duplex(char *apTxData, char *apRxData, long anLen);

private:
	std::string msDevice;
	int mnFileDescriptor;
};

#endif /* SPIDEVICE_H_ */
