/**
 * \file regs_upp.h
 *
 * \brief Register information for the uPP device.
 *
 *     o  0
 *     | /       Copyright (c) 2005-2010
 *    (CL)---o   Critical Link, LLC
 *      \
 *       O
 */

#ifndef _REGS_UPP_H_
#define _REGS_UPP_H_

#include <stdint.h>

////////////////////////////////////////////////////////////////////////////
/// @struct tsUppRegs tsUppRegs.h "core/tsUppRegs.h"
/// Private structure for uPP Control Registers. 
////////////////////////////////////////////////////////////////////////////
struct tsUppRegs
{
	volatile uint32_t UPPID; // 0x00 uPP Peripheral Identification Register Section 3.1
	volatile uint32_t UPPCR; // 0x04 uPP Peripheral Control Register Section 3.2
	volatile uint32_t UPDLB; // 0x08 uPP Digital Loopback Register Section 3.3
	volatile uint32_t EMPT0; // 0x0C
	volatile uint32_t UPCTL; // 0x10 uPP Channel Control Register Section 3.4
	volatile uint32_t UPICR; // 0x14 uPP Interface Configuration Register Section 3.5
	volatile uint32_t UPIVR; // 0x18 uPP Interface Idle Value Register Section 3.6
	volatile uint32_t UPTCR; // 0x1C uPP Threshold Configuration Register Section 3.7
	volatile uint32_t UPISR; // 0x20 uPP Interrupt Raw Status Register Section 3.8
	volatile uint32_t UPIER; // 0x24 uPP Interrupt Enabled Status Register Section 3.9
	volatile uint32_t UPIES; // 0x28 uPP Interrupt Enable Set Register Section 3.10
	volatile uint32_t UPIEC; // 0x2C uPP Interrupt Enable Clear Register Section 3.11
	volatile uint32_t UPEOI; // 0x30 uPP End-of-Interrupt Register Section 3.12
	volatile uint32_t EMPT1; // 0x34
	volatile uint32_t EMPT2; // 0x38
	volatile uint32_t EMPT3; // 0x3C
	volatile uint32_t UPID0; // 0x40 uPP DMA Channel I Descriptor 0 Register Section 3.13
	volatile uint32_t UPID1; // 0x44 uPP DMA Channel I Descriptor 1 Register Section 3.14
	volatile uint32_t UPID2; // 0x48 uPP DMA Channel I Descriptor 2 Register Section 3.15
	volatile uint32_t EMPT4; // 0x4C
	volatile uint32_t UPIS0; // 0x50 uPP DMA Channel I Status 0 Register Section 3.16
	volatile uint32_t UPIS1; // 0x54 uPP DMA Channel I Status 1 Register Section 3.17
	volatile uint32_t UPIS2; // 0x58 uPP DMA Channel I Status 2 Register Section 3.18
	volatile uint32_t EMPT5; // 0x5C
	volatile uint32_t UPQD0; // 0x60 uPP DMA Channel Q Descriptor 0 Register Section 3.19
	volatile uint32_t UPQD1; // 0x64 uPP DMA Channel Q Descriptor 1 Register Section 3.20
	volatile uint32_t UPQD2; // 0x68 uPP DMA Channel Q Descriptor 2 Register Section 3.21
	volatile uint32_t EMPT6; // 0x6C
	volatile uint32_t UPQS0; // 0x70 uPP DMA Channel Q Status 0 Register Section 3.22
	volatile uint32_t UPQS1; // 0x74 uPP DMA Channel Q Status 1 Register Section 3.23
	volatile uint32_t UPQS2; // 0x78 uPP DMA Channel Q Status 2 Register Section 3.24
};	

union tuUppcrReg
{
	uint32_t nRegWord;
	struct
	{
		uint32_t FREE	: 1;
		uint32_t SOFT	: 1;
		uint32_t RTEMU	: 1;
		uint32_t EN		: 1;
		uint32_t SWRST	: 1;
		uint32_t RSVD0	: 2;
		uint32_t DB		: 1;
		uint32_t RSVD1	: 24;
	} sRegBits;
};

union tuUpctlReg
{
	uint32_t nRegWord;
	struct
	{
		uint32_t MODE		: 2;
		uint32_t CHN		: 1;
		uint32_t SDRTXIL	: 1;
		uint32_t DDRDEMUX	: 1;
		uint32_t RSVD0		: 11;
		uint32_t DRA		: 1;
		uint32_t IWA		: 1;
		uint32_t DPWA		: 3;
		uint32_t DPFA		: 2;
		uint32_t RSVD1		: 1;
		uint32_t DRB		: 1;
		uint32_t IWB		: 1;
		uint32_t DPWB		: 3;
		uint32_t DPFB		: 2;
		uint32_t RSVD2		: 1;
	} sRegBits;
};

// MODE bits in UPCTL register.
enum teUppMode
{
	eeAllRcv 	= 0,
	eeAllXmit	= 1,
	eeARcvBXmit	= 2,
	eeAXmitBRcv	= 3
};

// Data Packing Format. DPFA and DPFB bits in UPCTL register.
enum teUppDpf
{
	eeRJZE	= 0,	// Right-justified, zero extended
	eeRJSE 	= 1,	// Right-justified, sign extended
	eeLJZF	= 2		// Left-justified, zero filled
};

union tuUpicrReg
{
	uint32_t nRegWord;
	struct
	{	
		uint32_t STARTPOLA	: 1;
		uint32_t ENAPOLA	: 1;
		uint32_t WAITPOLA	: 1;
		uint32_t STARTA		: 1;
		uint32_t ENAA		: 1;
		uint32_t WAITA		: 1;
		uint32_t RSVD0		: 2;
		uint32_t CLKDIVA	: 4;
		uint32_t CLKINVA	: 1;
		uint32_t TRISA		: 1;
		uint32_t RSVD1		: 2;
		uint32_t STARTPOLB	: 1;
		uint32_t ENAPOLB	: 1;
		uint32_t WAITPOLB	: 1;
		uint32_t STARTB		: 1;
		uint32_t ENAB		: 1;
		uint32_t WAITB		: 1;
		uint32_t RSVD2		: 2;
		uint32_t CLKDIVB	: 4;
		uint32_t CLKINVB	: 1;
		uint32_t TRISB		: 1;
		uint32_t RSVD3		: 2;
	} sRegBits;
};

union tuUpivrReg
{
	uint32_t nRegWord;
	struct
	{
		uint32_t VALA : 16;
		uint32_t VALB : 16;
	} sRegBits;
};

union tuUptcrReg
{
	uint32_t nRegWord;
	struct
	{
		uint32_t RDSIZEI 	: 2;
		uint32_t RSVD0	 	: 6;
		uint32_t RDSIZEQ	: 2;
		uint32_t RSVD1		: 6;
		uint32_t TXSIZEA	: 2;
		uint32_t RSVD2		: 6;
		uint32_t TXSIZEB	: 2;
		uint32_t RSVD3		: 6;
	} sRegBits;
};

union tuUpisrReg
{
	uint32_t nRegWord;
	struct
	{
		uint32_t DPEI	: 1;
		uint32_t UORI	: 1;
		uint32_t ERRI	: 1;
		uint32_t EOWI	: 1;
		uint32_t EOLI	: 1;
		uint32_t RSVD0	: 3;
		uint32_t DPEQ	: 1;
		uint32_t UORQ	: 1;
		uint32_t ERRQ	: 1;
		uint32_t EOWQ	: 1;
		uint32_t EOLQ	: 1;
		uint32_t RSVD1	: 19;
	} sRegBits;	
};

typedef tuUpisrReg tuUpierReg;
typedef tuUpisrReg tuUpiesReg;
typedef tuUpisrReg tuUpiecReg;

union tuUpiqd1Reg
{
	uint32_t nRegWord;
	struct
	{
		uint32_t BCNT	: 16;
		uint32_t LNCNT	: 16;	
	} sRegBits;
};

union tuUpiqs2Reg 
{
	uint32_t nRegWord;
	struct
	{
		uint32_t ACT	: 1;
		uint32_t PEND	: 1;
		uint32_t RSVD0	: 2;
		uint32_t WM	: 4;
		uint32_t RSVD1	: 24;
	} sRegBits;
};

#endif
