--- Title: spit.vhd
--- Description: Implementation of SPI (Serial Peripheral Interface) Core 
---
---     o  0
---     | /       Copyright (c) 2007-2010
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 08/11/2010
--- Version: 1.00
--- Revisions: 
---    1.00  Based loosely off of TI HPWM module

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
library UNISIM;
use UNISIM.VCOMPONENTS.ALL;
use work.MityDSP_L138_pkg.ALL;

entity pwm is
   Generic
   (
      NUM_OUTPUTS : integer := 2 -- MAX 4 unless you divide down the clock more
   );
   Port 
   (  
      emif_clk      : in  std_logic;
      i_ABus   : in std_logic_vector(5 downto 0);
      i_DBus   : in std_logic_vector(15 downto 0);
      o_DBus   :out std_logic_vector(15 downto 0);
      i_wr_en  : in std_logic;
      i_rd_en  : in std_logic;
      i_cs     : in std_logic;
      o_irq    :out std_logic := '0';
      i_ilevel       : in std_logic_vector(1 downto 0) := "00";      
      i_ivector      : in std_logic_vector(3 downto 0) := "0000";
      o_pwm          : out std_logic_vector(NUM_OUTPUTS-1 downto 0) := (others=>'0');
      o_sync         : out std_logic := '0';
      i_sync         : in  std_logic := '0'
    );                          
end pwm;						 

architecture rtl of pwm is

-- All Used components should be declared first.
--
constant CORE_VERSION_MAJOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 1, 4);
constant CORE_VERSION_MINOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 0, 4);
constant CORE_ID:             std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR( 8, 8);
constant CORE_YEAR:           std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 10, 5);
constant CORE_MONTH:          std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 11, 4);
constant CORE_DAY:            std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 22, 5);

signal version_reg       : std_logic_vector(15 downto 0) := (others=>'0');  -- version register
signal ver_rd            : std_logic := '0';
signal ctr_en : std_logic := '0';
signal clk_div : std_logic_vector(23 downto 0) := x"000100";
signal clk_rst_val : std_logic_vector(15 downto 0) := x"0000";
signal clk_prd : std_logic_vector(15 downto 0) := x"1000";
signal ram_a_out : std_logic_vector(15 downto 0) := (others=>'0');
signal ram_b_out : std_logic_vector(15 downto 0) := (others=>'0');
signal ram_a_addr : std_logic_vector(4 downto 0) := (others=>'0');
signal ram_b_addr : std_logic_vector(4 downto 0) := (others=>'0');
signal ram_a_we   : std_logic := '0';
signal div_cnt : std_logic_vector(23 downto 0) := (others=>'0');
signal clk_en  : std_logic := '0';
signal clk_cnt : std_logic_vector(15 downto 0) := (others=>'0');
signal pulse_count : std_logic_vector(2 downto 0) := (others=>'0');
signal check_zero, check_a, check_b : std_logic := '0';
signal pwm : std_logic_vector(NUM_OUTPUTS-1 downto 0) := (others=>'0');
signal ram_a_data : std_logic_vector(15 downto 0) := (others=>'0');
type   ram_type is array(31 downto 0) of std_logic_vector(15 downto 0);
signal ram :ram_type := (others=>(others=>'0'));
type   pulse_state is (CHECKA, CHECKB, SETPULSE, CHECKONESHOT);
signal sm_state : pulse_state := CHECKA;
signal one_shot : std_logic_vector(NUM_OUTPUTS-1 downto 0) := (others=>'0');
signal one_shot_clear : std_logic_vector(NUM_OUTPUTS-1 downto 0) := (others=>'0');
signal one_shot_clear_clear : std_logic_vector(NUM_OUTPUTS-1 downto 0) := (others=>'0');
signal active_ram : std_logic := '0';

begin

version : core_version
   port map(
      clk           => emif_clk,                  -- system clock
      rd            => ver_rd,               -- read enable
      ID            => CORE_ID,              -- assigned ID number, 0xFF if unassigned
      version_major => CORE_VERSION_MAJOR,   -- major version number 1-15
      version_minor => CORE_VERSION_MINOR,   -- minor version number 0-15
      year          => CORE_YEAR,            -- year since 2000
      month         => CORE_MONTH,           -- month (1-12)
      day           => CORE_DAY,             -- day (1-31)
      ilevel        => i_ilevel,
      ivector       => i_ivector,
      o_data        => version_reg
      );

-- This core doesn't emit any interrupts
o_irq <= '0';       

--* Clock in the transition conditions for each pulse, which is 
--* a set of 3 16 bit values per pulse, as follows:
--* A Action count (the clk Count when A Action occurs)
--* B Action count (the clk Count when B Action occurs)
--* Control Bits as follows(LSB first):
--*		zero_action | 2 Bits | 00=ignore,01=set zero,10=set one,11=toggle		
--*		a_action 	| 2 Bits | 00=ignore,01=set zero,10=set one,11=toggle		
--*		b_action 	| 2 Bits | 00=ignore,01=set zero,10=set one,11=toggle		
--*		force_one 	| 1 Bits | 0=ignore, 1=set one
--*		force_zero	| 1 Bits | 0=ignore, 1=set zero
--*		unused 		| 1 Bits | 
--*		1shot_action| 2 Bits | 00=ignore,01=set zero,10=set one,11=toggle		
--*		unused 		| 4 Bits | 
--/
process(emif_clk)
begin
   if rising_edge(emif_clk) then
      if ram_a_we='1' then
         ram(conv_integer(ram_a_addr)) <= ram_a_data;
      end if;
      ram_a_out <= ram(conv_integer(ram_a_addr));
   end if;      
end process;

ram_b_out <= ram(conv_integer(ram_b_addr));
                    
--* Handle read requests from the processor
--/
read_mux_regs : process (emif_clk) is
begin
	if emif_clk'event and emif_clk='1' then

      ver_rd <= '0';

      -- address decoding
      if i_cs = '0' then
         o_DBus <= (others=>'0');
      else
         case i_ABus is
            when "000000" =>   
               o_DBus <= version_reg;
               ver_rd <= i_rd_en;
            when "000001" =>
               o_DBus <= clk_div(11 downto 0) & '0' & '0' & active_ram & ctr_en;	
            when "000010" =>
               o_DBus <= x"0" & clk_div(23 downto 12);	
   			when "000011" =>	
   			   o_DBus <= clk_rst_val;
   			when "000100" =>	
   			   o_DBus <= clk_prd;
            when "000101" =>	
               o_DBus <= x"00" & "000" & ram_a_addr;               
            when "000110" =>	
               o_DBus <= ram_a_out;  
            when "000111" =>
               o_DBus <= ext(one_shot,8) & ext(one_shot_clear,8);             
   			when others =>	
               o_DBus <= (others=>'0');
         end case;
      end if;
   end if;
end process read_mux_regs;

--* Decode register write requests.
--/
wr_ctl_reg : process(emif_clk)
begin
	if emif_clk='1' and emif_clk'event then
        ram_a_we <= '0';
               
		if i_cs = '1' and i_wr_en = '1' then
			case i_ABus is
			    when "000001" => 
			       ctr_en <= i_DBus(0);
			       active_ram <= i_DBus(1);
			       clk_div(11 downto 0) <= i_DBus(15 downto 8) & "0000";
       			when "000010" =>  
       			   clk_div(23 downto 12) <= i_DBus(11 downto 0);
       			when "000011" =>  
       			   clk_rst_val <= i_DBus;
       			when "000100" =>
       			   clk_prd <= i_DBus;
       			when "000101" =>
       			   ram_a_addr <= i_DBus(4 downto 0);
    			when "000110" => 
    			   ram_a_we <= '1'; 
    			   ram_a_data <= i_DBus;
                when "000111" =>
                   one_shot_clear <= i_DBus(NUM_OUTPUTS-1 downto 0);            
    			when others => NULL;
			end case;
	   else
	        for i in 0 to NUM_OUTPUTS-1 loop
	            if one_shot_clear_clear(i)='1' then
	               one_shot_clear(i) <= '0';
	            end if;
	        end loop;
	   end if;	       
	end if;
end process wr_ctl_reg;

clk_divide : process(emif_clk)
begin
   if rising_edge(emif_clk) then
       if div_cnt >= clk_div then
          div_cnt <= (others=>'0');
          clk_en <= '1';
       else
          div_cnt <= div_cnt+'1';
          clk_en <= '0';
       end if;
   end if;
end process clk_divide;

-- Counts the number of clock divided
-- Clock ticks, resets when period is hit
clk_count : process(emif_clk)
begin
   if rising_edge(emif_clk) then
      if clk_en='1' then
         -- external sync
         if i_sync='1' then
            clk_cnt <= clk_rst_val;
         -- disabled
         elsif ctr_en='0' then
            clk_cnt <= (others=>'0');
         else
            if clk_cnt >= clk_prd then
               clk_cnt <= (others=>'0');
            else
               clk_cnt <= clk_cnt+'1';
            end if;
         end if;
      end if;
   end if;
end process clk_count;

outputs : process(emif_clk)
begin
   if rising_edge(emif_clk) then
      if clk_en='1' then
		  -- This is the end of a clock period
		  -- reset our pulse index back to 0
          pulse_count <= CONV_STD_LOGIC_VECTOR(0,3);
		  -- Output pulses
          o_pwm <= pwm;
		  -- Check if the clock count is reset
          if clk_cnt = CONV_STD_LOGIC_VECTOR(0,16) then
              check_zero <= '1';
          else
            check_zero <= '0';
          end if;
		  -- Reset our ram index
          ram_b_addr <= active_ram & "0000";
      end if;
      if pulse_count /= CONV_STD_LOGIC_VECTOR(NUM_OUTPUTS,3) then
          case sm_state is
             when CHECKA =>
			 	-- See if our current clk count matches what
				-- was stored in RAM for A Action
                if ram_b_out = clk_cnt then
                    check_a <= '1';
                else
                    check_a <= '0';
                end if;
                sm_state <= CHECKB;
				-- Move ram index to B Action count
                ram_b_addr <= ram_b_addr+'1';
             when CHECKB =>
			 	-- See if our current clk count matches what
				-- was stored in RAM for B Action
                if ram_b_out = clk_cnt then
                    check_b <= '1';
                else
                    check_b <= '0';
                end if;
                sm_state <= SETPULSE;
				-- Move ram index to control vector for this Action
                ram_b_addr <= ram_b_addr+'1';
            when SETPULSE =>
                -- force on
                if ram_b_out(7) = '1' then
                   pwm(CONV_INTEGER(pulse_count)) <= '1';
                -- force off
                elsif ram_b_out(6) = '1' then
                   pwm(CONV_INTEGER(pulse_count)) <= '0';
                -- reg B event
                elsif check_b='1' and ram_b_out(5 downto 4) /= "00" then
                       case ram_b_out(5 downto 4) is
                          when "01" => pwm(CONV_INTEGER(pulse_count)) <= '0';
                          when "10" => pwm(CONV_INTEGER(pulse_count)) <= '1';
                          when "11" => pwm(CONV_INTEGER(pulse_count)) <= not pwm(CONV_INTEGER(pulse_count));
                          when others => NULL;
                       end case;
                -- reg A event                       
                elsif check_a='1' and ram_b_out(3 downto 2) /= "00" then
                       case ram_b_out(3 downto 2) is
                          when "01" => pwm(CONV_INTEGER(pulse_count)) <= '0';
                          when "10" => pwm(CONV_INTEGER(pulse_count)) <= '1';
                          when "11" => pwm(CONV_INTEGER(pulse_count)) <= not pwm(CONV_INTEGER(pulse_count));
                          when others => NULL;
                       end case;     
                -- reset event                  
                elsif check_zero='1' and ram_b_out(1 downto 0) /= "00" then
                       case ram_b_out(1 downto 0) is
                          when "01" => pwm(CONV_INTEGER(pulse_count)) <= '0';
                          when "10" => pwm(CONV_INTEGER(pulse_count)) <= '1';
                          when "11" => pwm(CONV_INTEGER(pulse_count)) <= not pwm(CONV_INTEGER(pulse_count));
                          when others => NULL;
                      end case;                       
                end if;
                sm_state <= CHECKONESHOT;
            when CHECKONESHOT =>
                case ram_b_out(11 downto 10) is
                    when "01" =>
                        if check_zero='1' then
                           if one_shot_clear(CONV_INTEGER(pulse_count))='1' then
                               one_shot(CONV_INTEGER(pulse_count)) <= '0';
                               one_shot_clear_clear(CONV_INTEGER(pulse_count)) <= '1';
                           else
                               one_shot(CONV_INTEGER(pulse_count)) <= '1';
                               one_shot_clear_clear(CONV_INTEGER(pulse_count)) <= '0';
                           end if;
                        end if;
                    when "10" =>
                        if check_a='1' then
                           if one_shot_clear(CONV_INTEGER(pulse_count))='1' then
                               one_shot(CONV_INTEGER(pulse_count)) <= '0';
                               one_shot_clear_clear(CONV_INTEGER(pulse_count)) <= '1';
                           else
                               one_shot(CONV_INTEGER(pulse_count)) <= '1';
                               one_shot_clear_clear(CONV_INTEGER(pulse_count)) <= '0';
                           end if;
                        end if;
                    when "11" =>
                        if check_b='1' then
                           if one_shot_clear(CONV_INTEGER(pulse_count))='1' then
                               one_shot(CONV_INTEGER(pulse_count)) <= '0';
                               one_shot_clear_clear(CONV_INTEGER(pulse_count)) <= '1';
                           else
                               one_shot_clear_clear(CONV_INTEGER(pulse_count)) <= '0';
                               one_shot(CONV_INTEGER(pulse_count)) <= '1';
                           end if;
                        end if;
                    when others => NULL;
                end case;
                if one_shot(CONV_INTEGER(pulse_count))='1' and ram_b_out(11 downto 10)/="00" then
                    pwm(CONV_INTEGER(pulse_count)) <= ram_b_out(8);
                end if;
                sm_state <= CHECKA;
                pulse_count <= pulse_count+'1';
                ram_b_addr <= ram_b_addr+'1';
            when OTHERS => NULL;
         end case;
      end if;
   end if;    
end process outputs;

end rtl;
