--- Title: ads7843.vhd
--- Description: Touch Screen Interface Controller (ADS7843)
---
---     o  0
---     | /       Copyright (c) 2007
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 03/19/2007
--- Version: 1.00
--- Revisions:
--- 1.00 2007/03/28 Initial version - PenIRQ not functioning
--- 1.00 2007/06/01 PenIRQ Fixed - fully functional with the SW
--- 1.01 2009/03/20 Slowed DCLK down to 100 KHz based on 50 MHz EMIF clock
--- 1.02 2010/02/03 Add support for GPIO and Disp I/O screen interface

library STD;
use STD.TEXTIO.ALL;
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use IEEE.STD_LOGIC_TEXTIO.ALL;

library WORK;
use WORK.MityDSP_L138_pkg.ALL;

entity ads7843 is
   port (
      emif_clk       : in std_logic; -- 100 Mhz EMIF clock
      i_ABus         : in  std_logic_vector(5 downto 0);
      i_DBus         : in  std_logic_vector(15 downto 0);
      o_DBus         : out std_logic_vector(15 downto 0);
      i_wr_en        : in  std_logic;
      i_rd_en        : in  std_logic;
      i_cs           : in  std_logic;	
      o_irq          : out std_logic;
      i_ilevel       : in std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      i_ivector      : in std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)

      o_ts_cs_n      :out std_logic;
      o_ts_clk       :out std_logic;
      o_ts_din       :out std_logic;
      i_ts_dout      : in std_logic;
      i_ts_busy      : in std_logic;
      i_ts_PenIRQ_n  : in std_logic;
      o_ts_PenIRQ_n  :out std_logic;
      t_ts_PenIRQ_n  :out std_logic;
      o_disp_cs_n    :out std_logic;
      o_gpio_out     :out std_logic_vector(23 downto 0)
   );
end ads7843;

--------------------------------------------------------------------------
-- ARCHITECTURE
--------------------------------------------------------------------------
architecture rtl of ads7843 is

constant CORE_APPLICATION_ID: std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR( 23, 8);
constant CORE_VERSION_MAJOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 1, 4);
constant CORE_VERSION_MINOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 2, 4);
constant CORE_YEAR:           std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 10, 5);
constant CORE_MONTH:          std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 02, 4);
constant CORE_DAY:            std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 15, 5);

constant FE : integer := 15*16;  --TS_CLK Falling Edge  (slow down the clock)
constant RE : integer := 31*16;  --TS_CLK Rising Edge   (slow down the clock)

signal version_reg       : std_logic_vector(15 downto 0);  -- version register
signal ver_rd            : std_logic;
signal irq_en, clear_flag, new_coord, disp_irq_en : std_logic := '0';

--attribute SIGNAL_ENCODING      : string;
type ts_state_type is (TS_WAITPEN, TS_GETX, TS_GETY, TS_POWER_DOWN, DISP_CMD, DISP_GPIO);
--attribute SIGNAL_ENCODING of ts_state_type: type is "001 010 100";
signal ts_cs, ts_ns : ts_state_type := TS_WAITPEN;

signal ts_dout_meta : std_logic := '0';
signal data21 : std_logic_vector(23 downto 0) := x"000000";    --Data2 & Data1
signal datax, datay : std_logic_vector(11 downto 0) := "000000000000";
signal datax_reg, datay_reg : std_logic_vector(11 downto 0) := "000000000000";
signal dataxcmp_ping, dataycmp_ping : std_logic_vector(11 downto 0) := x"000";
signal dataxcmp_pong : std_logic_vector(11 downto 0) := x"000";
signal dataycmp_pong : std_logic_vector(11 downto 0) := x"000";
signal ping_pong : std_logic := '1';
signal xmatch, xmatch_r, ymatch, ymatch_r : std_logic := '0';
signal ts_clk, ts_penIRQ_meta, ts_penIRQ : std_logic := '0'; 
signal read_pen, in_use : std_logic := '0';
signal ts_cs_n : std_logic := '1';
signal disp_cs_n : std_logic := '1';
signal time_delay_sh : std_logic_vector(15 downto 0) := x"0001";  -- ~250Hz sampling

signal data_zero, data_zero_r : std_logic := '0';
signal xy_inrange : std_logic := '0';

--Clock generated from a shift register
signal ts_clk_sh : std_logic_vector(RE downto 0) := (0=>'1', others=>'0');
signal dst_sh : std_logic_vector(23 downto 0) := x"000001";    --Data State

--Dout Shift Register: Start | A2 | A1 | A0 | 12-bit# | SER/DIF# | PD1 | PD0
--  X address : 001
--  Y address : 101
--  Power Down (PD1:0)
--    - 00 = Power Down Between Conversions w/ PenIRQ enabled
--    - 11 = No Power Down Between Conversions
--signal dosh_xpd00 : std_lgoic_vector(7 downto 0) := x"94";
--signal dosh_xpd00 : std_logic_vector(7 downto 0) := x"94";
--signal dosh_xpd11 : std_logic_vector(7 downto 0) := x"97";  --Single Ended
--signal dosh_ypd11 : std_logic_vector(7 downto 0) := x"D7";  --Single Ended
signal dosh_xpd11 : std_logic_vector(7 downto 0) := x"93";  --Differential
signal dosh_ypd00 : std_logic_vector(7 downto 0) := x"D0";  --Differential
signal dosh_ypd11 : std_logic_vector(7 downto 0) := x"D3";  --Differential

signal ctl_mask, data_mask : std_logic := '0';
signal no_bounce : std_logic := '0';
signal ts_init : std_logic := '1';
signal conv_cnt : std_logic_vector(3 downto 0) := "0000";
signal data_timeout : std_logic := '0';

signal disp_cmd_written, disp_cmd_written_r1 : std_logic := '0';
signal gpio_cycle_en : std_logic := '0';
signal gpio_dat, cmd_resp, gpio_cmd, dsp_cmd, dosh_spi_cmd : std_logic_vector(23 downto 0) := x"000000";
signal cmd_done : std_logic := '0';

signal s_i_ts_busy : std_logic := '0';

begin -- architecture: rtl

version : core_version
   port map(
      clk           => emif_clk,             -- system clock
      rd            => ver_rd,               -- read enable
      ID            => CORE_APPLICATION_ID,  -- assigned ID number, 0xFF if unassigned
      version_major => CORE_VERSION_MAJOR,   -- major version number 1-15
      version_minor => CORE_VERSION_MINOR,   -- minor version number 0-15
      year          => CORE_YEAR,            -- year since 2000
      month         => CORE_MONTH,           -- month (1-12)
      day           => CORE_DAY,             -- day (1-31)
      ilevel        => i_ilevel,
      ivector       => i_ivector,
      o_data        => version_reg );

o_ts_cs_n <= ts_cs_n;
o_ts_PenIRQ_n <= '0';
t_ts_PenIRQ_n <= read_pen;
o_disp_cs_n <= disp_cs_n;
o_gpio_out  <= gpio_dat;

o_ts_clk <= ts_clk;

--* Handle read requests from the processor
--/
read_mux_regs : process (emif_clk) is
begin
	if emif_clk'event and emif_clk='1' then
      o_irq <= (irq_en and new_coord) or (disp_irq_en and cmd_done);
      
      disp_cmd_written_r1 <= disp_cmd_written;

      -- toggle version register
      if i_ABus="000000" and i_cs='1' and i_rd_en='1' then
         ver_rd <= '1';
      else
         ver_rd <= '0';
      end if;
      
      if i_ABus="000111" and i_cs='1' and i_rd_en='1' then
         cmd_done <= '0';
      elsif disp_cmd_written = '0' and disp_cmd_written_r1 = '1' then
         cmd_done <= '1';
      end if;
     
	s_i_ts_busy <= i_ts_busy;

      -- address decoding
      if i_cs = '0' then
         o_DBus <= (others=>'0');
      else
         case i_ABus is
            when "000000" =>
               o_DBus <=  version_reg;
            when "000010" =>
               o_DBus <= disp_irq_en & cmd_done & gpio_cycle_en & xy_inrange & datay_reg;
            when "000011" =>
               o_DBus <= irq_en & new_coord & in_use & s_i_ts_busy & datax_reg;
            when "000110" =>
               o_DBus <= cmd_resp(15 downto 0);
            when "000111" =>
               o_DBus(15 downto 8) <= (others=>'0');
               o_DBus(7 downto 0) <= cmd_resp(23 downto 16);
            when "001000" =>
               o_DBus <= gpio_dat(15 downto 0);
            when "001001" =>
               o_DBus(15 downto 8) <= (others=>'0');
               o_DBus(7 downto 0) <= gpio_dat(23 downto 16);
            when others =>
               o_DBus <= (others=>'0');
         end case;
      end if;
   end if;
end process read_mux_regs;

--* Decode register write requests.
--/
wr_ctl_reg : process(emif_clk)
begin
   if emif_clk='1' and emif_clk'event then
      clear_flag <= '0';
      
      if i_cs = '1' and i_wr_en = '1' and i_ABus = "000010" then
         ts_init <= i_DBus(15);
      elsif ts_cs = TS_POWER_DOWN then
         ts_init <= '0';   --A reset to enable the power-down mode w/ PenIRQ enabled
      end if;
      
      if i_cs = '1' and i_wr_en = '1' and i_ABus = "000101" then
         disp_cmd_written <= '1';
      elsif ts_cs = DISP_CMD and dst_sh(23) = '1' and ts_clk_sh(RE) = '1' then
         disp_cmd_written <= '0';
      end if;
      
      if i_cs = '1' and i_wr_en = '1' then
         case i_ABus is
            when "000010" =>   
               disp_irq_en <= i_DBus(15);
               gpio_cycle_en <= i_DBus(13);
            when "000011" =>   
               irq_en <= i_DBus(15);
               clear_flag <= i_DBus(14);     --Single clk pulse to clear NewData_Flag
            when "000100" =>
               dsp_cmd(15 downto 0) <= i_DBus;
            when "000101" =>
               dsp_cmd(23 downto 16) <= i_DBus(7 downto 0);
            when "001000" =>
               gpio_cmd(15 downto 0) <= i_DBus;
            when "001001" =>
               gpio_cmd(23 downto 16) <= i_DBus(7 downto 0);
            when others => NULL;
         end case;
      end if;
   end if;
end process wr_ctl_reg;

ts_ctl : process(emif_clk)
begin
   if emif_clk'event and emif_clk='1' then
      ts_dout_meta <= i_ts_dout;
      --Divide the clock to 1.5MHz or lower
      --  RE = Rising Edge Bit
      --  FE = Falling Edge Bit
      ts_clk_sh <= ts_clk_sh(RE-1 downto 0) & ts_clk_sh(RE);

      --Data State Shift Register
      --    8 bits of command, 1 busy, 12 data, 3 unused = 24 data bits to shift
      if ts_clk_sh(RE) = '1' then
         dst_sh <= dst_sh(22 downto 0) & dst_sh(23);
      end if;
      if dst_sh(8) = '1' then
         ctl_mask <= '0';
      elsif dst_sh(0) = '1' and ts_cs /= TS_WAITPEN then
         ctl_mask <= '1';
      end if;
      if dst_sh(21) = '1' then
         data_mask <= '0';
      elsif dst_sh(9) = '1' and (ts_cs = TS_GETX or ts_cs = TS_GETY) then
         data_mask <= '1';
      end if;

      -- disable the touch screen chip select when waiting for the pen (this 
      -- includes the periods when shifting data to the display SPI lines...)
      if ts_cs = TS_WAITPEN and time_delay_sh(15) = '1' then 
         ts_cs_n <= '0';
      elsif ts_cs = TS_WAITPEN and dst_sh(8) = '1' then
         ts_cs_n <= '1';
      end if;
      
      -- enable the display chip select appropriately
      if (ts_cs = DISP_CMD or ts_cs = DISP_GPIO) and ts_clk_sh(FE) = '1' then
         disp_cs_n <= '0';
      elsif ts_clk_sh(FE) = '1' then
         disp_cs_n <= '1';
      end if;
      
      if dst_sh(23) = '1' and ts_clk_sh(RE) = '1' then
         time_delay_sh <= time_delay_sh(14 downto 0) & time_delay_sh(15);
      end if;

      --Wait for penIRQ
      if ts_cs = TS_WAITPEN then
         read_pen <= '1';
      else
         read_pen <= '0';
      end if;

      ts_penIRQ_meta <= not i_ts_PenIRQ_n;
      if ts_clk_sh(RE) = '1' and read_pen = '1' and dst_sh(21) = '1' then
         ts_penIRQ <= ts_penIRQ_meta;
      end if;

      if ts_cs = TS_WAITPEN and dst_sh(22) = '1' and ts_clk_sh(RE) = '1' then
         in_use <= ts_penIRQ;
      end if;

      if ts_clk_sh(FE) = '1' then      --Falling Edge of ts_clk
         ts_clk <= '0';
      elsif ts_clk_sh(RE) = '1' then   --Rising edge of ts_clk
         ts_clk <= '1';
      end if;

      if time_delay_sh(0) = '1' then
         dosh_spi_cmd <= dsp_cmd;
      elsif time_delay_sh(6) = '1' then
         dosh_spi_cmd <= gpio_cmd;
      elsif ts_clk_sh(FE) = '1' and (ts_cs = DISP_CMD or ts_cs = DISP_GPIO) then
         dosh_spi_cmd <= dosh_spi_cmd(22 downto 0) & '0';
      end if;

      --Clock falling edge events
      --Data output changes
      if ts_clk_sh(FE) = '1' then   --Falling edge of ts_clk
         dosh_xpd11 <= dosh_xpd11(6 downto 0) & dosh_xpd11(7); --No Power Down
         dosh_ypd11 <= dosh_ypd11(6 downto 0) & dosh_ypd11(7); --No Power Down
         dosh_ypd00 <= dosh_ypd00(6 downto 0) & dosh_ypd00(7); --Power Down, PenIRQ Enabled
         

         o_ts_din <= '0';     --Default value
         if ts_cs = DISP_CMD then
            o_ts_din <= dosh_spi_cmd(23);
         elsif ts_cs = DISP_GPIO then
            o_ts_din <= dosh_spi_cmd(23);
         -- touch screen shiftout logic starts here....         
         elsif ctl_mask = '0' then
            o_ts_din <= '0';
         elsif ts_cs = TS_GETX then
            o_ts_din <= dosh_xpd11(7);    --DataOutShifter
         elsif ts_cs = TS_GETY then
            o_ts_din <= dosh_ypd11(7);
         elsif ts_cs = TS_POWER_DOWN then
            o_ts_din <= dosh_ypd00(7);
         end if;
      end if;

      --Clock Rising edge events...
      --Clock in the data (Data2 & Data1)
      if ts_clk_sh(RE) = '1' then
         if data_mask = '1' or ts_cs = DISP_GPIO or ts_cs = DISP_CMD then
            data21 <= data21(22 downto 0) & ts_dout_meta;
         end if;
         
         if dst_sh(23) = '1' and ts_cs = DISP_GPIO then
            gpio_dat <= data21(22 downto 0) & ts_dout_meta;
         end if;
         
         if dst_sh(23) = '1' and ts_cs = DISP_CMD then
            cmd_resp <= data21(22 downto 0) & ts_dout_meta;
         end if;

         --Temporary register for X value - only given to the DSP if both X&Y good
         if dst_sh(21) = '1' and no_bounce = '1' then
            if ts_cs = TS_GETX then
               datax <= data21(11 downto 0);
            end if;
         end if;

         --Register the data values for the DSP if touched
         if dst_sh(23) = '1' and ts_cs = TS_GETY and no_bounce = '1' and new_coord = '1' then
            datax_reg <= datax;
            datay_reg <= data21(11 downto 0);      --datay;
         end if;

         --Compare shift registers
         --  Update every other coordinate read to check against a static previous XY
         if data_mask = '1' then
            --Shifting data to compare against
            dataxcmp_ping <= dataxcmp_ping(10 downto 0) & dataxcmp_ping(11);
            dataycmp_ping <= dataycmp_ping(10 downto 0) & dataycmp_ping(11);
            --New data shifted in if no match
            if ping_pong = '1' and ts_cs = TS_GETX then
               dataxcmp_ping(0) <= data21(11);
            end if;
            if ping_pong = '1' and ts_cs = TS_GETY then
               dataycmp_ping(0) <= data21(11);
            end if;
            --Shifting data to compare against
            dataxcmp_pong <= dataxcmp_pong(10 downto 0) & dataxcmp_pong(11);
            dataycmp_pong <= dataycmp_pong(10 downto 0) & dataycmp_pong(11);
            --New data shifted in if no match
            if ping_pong = '0' and ts_cs = TS_GETX then
               dataxcmp_pong(0) <= data21(11);
            end if;
            if ping_pong = '0' and ts_cs = TS_GETY then
               dataycmp_pong(0) <= data21(11);
            end if;
         end if;
         --PenIRQ input not working - check for valid range
         if dst_sh(23) = '1' then
            data_zero <= '1';
         elsif data_mask = '1' and data21(11) = '1' then
            data_zero <= '0';
         end if;
         if dst_sh(20) = '1' and ts_cs = TS_GETX then       --19 ignores bit 0
            data_zero_r <= data_zero;
            data_timeout <= conv_cnt(3);
         elsif dst_sh(20) = '1' and ts_cs = TS_GETY then    --19 ignores bit 0
            data_zero_r <= data_zero or data_zero_r;
            data_timeout <= data_timeout or conv_cnt(3);
         elsif dst_sh(20) = '1' then
            data_timeout <= '0';
         end if;

         --Check for data change
         if ts_cs /= ts_ns
         or (ping_pong = '1' and data_mask = '1' and data21(11) /= dataxcmp_pong(11))
         or (ping_pong = '0' and data_mask = '1' and data21(11) /= dataxcmp_ping(11)) then
            xmatch <= '0';
         elsif dst_sh(23) = '1' then
            xmatch <= '1';
         end if;
         if ts_cs /= ts_ns
         or (ping_pong = '1' and data_mask = '1' and data21(11) /= dataycmp_pong(11))
         or (ping_pong = '0' and data_mask = '1' and data21(11) /= dataycmp_ping(11)) then
            ymatch <= '0';
         elsif dst_sh(23) = '1' then
            ymatch <= '1';
         end if;
         --Register if the X coordinate matches the previous X coord
         if dst_sh(21) = '1' and ts_cs = TS_GETX then
            xmatch_r <= xmatch;
         end if;
         --Register if the Y coordinate matches the previous Y coord
         if dst_sh(21) = '1' and ts_cs = TS_GETY then
            ymatch_r <= ymatch;
         end if;
         if dst_sh(21) = '1' and ts_cs = TS_POWER_DOWN then
            ping_pong <= not ping_pong;
         end if;
      end if;

      --Debounce the sampling...  Keep sampling until data is same for 2 samples (or give up)
      if (data_mask = '1' and data21(11) /= ts_dout_meta and ts_clk_sh(RE) = '1')
      or ts_cs /= ts_ns then
         no_bounce <= '0';
      elsif dst_sh(23) = '1' and ts_clk_sh(RE) = '1' then
         no_bounce <= '1';
      end if;

      --Can't debounce the sample when pen released...  Timeout the # of conversions
      --  Assume the data will settle within 7 cycles or abort the conversion
      if ts_cs /= ts_ns then
         conv_cnt <= "0000";
      elsif dst_sh(23) = '1' and ts_clk_sh(RE) = '1' then
         conv_cnt <= conv_cnt + '1';
      end if;

      --IRQ Flag when new data point
      if clear_flag = '1' then
         new_coord <= '0';
      elsif ts_cs = TS_GETY and no_bounce = '1' and dst_sh(22) = '1'
      and (xmatch_r = '0' or ymatch_r = '0') and data_zero_r = '0'
      and ts_penIRQ = '1' and data_timeout = '0' then
         new_coord <= '1';
      end if;
      if ts_cs = TS_GETY and no_bounce = '1' and dst_sh(22) = '1' then
         xy_inrange <= not data_zero_r;
      end if;
   end if;
end process;

--Sync State Advance
adv_st : process(emif_clk)
begin
   if emif_clk'event and emif_clk = '1' then
      ts_cs <= ts_ns;
   end if;
end process;

--Async Touchscreen State Machine
ts_sm : process(ts_cs, dst_sh, ts_PenIRQ, no_bounce, time_delay_sh, ts_clk_sh, ts_init)
begin
   --Default state to avoid latches
   ts_ns <= TS_WAITPEN;

   case ts_cs is
      --IDLE state, normally waiting for the Pen input
      when TS_WAITPEN =>
         -- start of a new 24 bit clock out cycle
         if dst_sh(23) = '1' and ts_clk_sh(RE) = '1' then
             -- if a pen interrupt is seen, start the readout cycle (but only after a reasonable delay)
             if time_delay_sh(15) = '1' and ts_PenIRQ = '1' then
                ts_ns <= TS_GETX;
             -- PowerDown with PenIRQ Enabled periodically for startup case
             elsif time_delay_sh(15) = '1' and ts_PenIRQ = '0' and ts_init = '1' then
                ts_ns <= TS_POWER_DOWN;
             -- display command is pending and we are at the start of a cycle
             elsif time_delay_sh(1) = '1' and disp_cmd_written = '1' then
                ts_ns <= DISP_CMD;
             -- insert a GPIO cycle if enabled
             elsif time_delay_sh(7) = '1' and gpio_cycle_en = '1' then
                ts_ns <= DISP_GPIO;
             -- nothing else going on, stay in WAITPEN
             else
                ts_ns <= TS_WAITPEN;
             end if;
         else
             ts_ns <= TS_WAITPEN;
         end if;
      --Get the X sample
      when TS_GETX =>
         if (dst_sh(23) = '1' and ts_clk_sh(RE) = '1')      --Conv Resolved
         and (no_bounce = '1' or data_timeout = '1') then   --...or Give Up
            ts_ns <= TS_GETY;
         else
            ts_ns <= TS_GETX;
         end if;
      --Get the Y sample
      when TS_GETY =>
         if (dst_sh(23) = '1' and ts_clk_sh(RE) = '1')      --Conv Resolved
         and (no_bounce = '1' or data_timeout = '1') then   --...or Give Up
            ts_ns <= TS_POWER_DOWN;
         else
            ts_ns <= TS_GETY;
         end if;
      when TS_POWER_DOWN =>
         if dst_sh(23) = '1' and ts_clk_sh(RE) = '1' then
            ts_ns <= TS_WAITPEN;
         else
            ts_ns <= TS_POWER_DOWN;
         end if;
      when DISP_CMD =>
         if dst_sh(23) = '1' and ts_clk_sh(RE) = '1' then
            ts_ns <= TS_WAITPEN;
         else
            ts_ns <= DISP_CMD;
         end if;
      when DISP_GPIO =>
         if dst_sh(23) = '1' and ts_clk_sh(RE) = '1' then
            ts_ns <= TS_WAITPEN;
         else
            ts_ns <= DISP_GPIO;
         end if;
      when others => ts_ns <= TS_WAITPEN;
   end case;
end process;

end rtl;
