--- Title: timing_gen_ext.vhd
--- Description: This is a wrapper for the Timing Generator Core 
---              Block RAM size can be selectable to use 512 or 1024   
---
---
---
---     o  0                          
---     | /       Copyright (c) 2009  
---    (CL)---o   Critical Link, LLC  
---      \                            
---       O                           
---
--- Company: Critical Link, LLC
--- Date: 05/21/2009
--- Version: 
---   1.00 - 05/21/2009 - ZNT - Initial Version

library IEEE;
use     IEEE.STD_LOGIC_1164.ALL;
use     IEEE.STD_LOGIC_ARITH.ALL;
use     IEEE.STD_LOGIC_UNSIGNED.ALL;
library UNISIM;
use     UNISIM.VCOMPONENTS.all;

entity timing_gen is
   generic
   (
      BRAM_DEPTH_TWO_TO_N : integer range 9 to 10 := 10 --  9 is  512 x 32
   );                                                   -- 10 is 1024 x 32
   Port ( 
      -- EMIF Interface signals
      i_emif_clk : in  std_logic; -- emif_clk (system clock) from top-level
      i_ABus     : in  std_logic_vector(5 downto 0);
      i_DBus     : in  std_logic_vector(15 downto 0);
      o_DBus     : out std_logic_vector(15 downto 0);
      i_wr_en    : in  std_logic;
      i_rd_en    : in  std_logic;
      i_cs       : in  std_logic;
      -- IRQ system signals
      o_irq      : out std_logic;
      i_ilevel       : in    std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      i_ivector      : in    std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)
		-- Timing Generator I/O 
      i_clk      : in  std_logic;
      i_clk_en   : in  std_logic;
		i_wait_inp : in std_logic_vector(15 downto 0);
      o_output   : out std_logic_vector(15 downto 0)
    );
end timing_gen;


architecture rtl of timing_gen is

--
-- All Used components should be declared first
--

component timing_gen_ext
   generic
   (
      BRAM_DEPTH_TWO_TO_N : integer range 9 to 10 := 10 --  9 is  512 x 32
   );                                                   -- 10 is 1024 x 32
   port ( 
      -- EMIF Interface signals
      i_emif_clk : in  std_logic; -- emif_clk (system clock) from top-level
      i_ABus     : in  std_logic_vector(5 downto 0);
      i_DBus     : in  std_logic_vector(15 downto 0);
      o_DBus     : out std_logic_vector(15 downto 0);
      i_wr_en    : in  std_logic;
      i_rd_en    : in  std_logic;
      i_cs       : in  std_logic;
      -- IRQ system signals
      o_irq      : out std_logic;
      i_ilevel       : in    std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      i_ivector      : in    std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)

		-- External 32-bit Opcode Write Ports
		o_clka  : out std_logic;
		o_addra : out std_logic_vector(BRAM_DEPTH_TWO_TO_N-1 downto 0);
		o_dina  : out std_logic_vector(31 downto 0);
		i_douta : in  std_logic_vector(31 downto 0);
		o_wea   : out std_logic;

		-- External 32-bit Opcode Read Ports
		o_clkb  : out std_logic;
		o_addrb : out std_logic_vector(BRAM_DEPTH_TWO_TO_N-1 downto 0);
		i_doutb : in  std_logic_vector(31 downto 0);
		o_enb   : out std_logic;

		-- Timing Generator I/O 
      i_clk      : in  std_logic;
      i_clk_en   : in  std_logic;
		i_wait_inp : in std_logic_vector(15 downto 0);
      o_output   : out std_logic_vector(15 downto 0)
    );
end component;

COMPONENT timing_gen_dpbram_512x32
  PORT (
    clka : IN STD_LOGIC;
    wea : IN STD_LOGIC_VECTOR(0 DOWNTO 0);
    addra : IN STD_LOGIC_VECTOR(8 DOWNTO 0);
    dina : IN STD_LOGIC_VECTOR(31 DOWNTO 0);
    douta : OUT STD_LOGIC_VECTOR(31 DOWNTO 0);
    clkb : IN STD_LOGIC;
    enb : IN STD_LOGIC;
    web : IN STD_LOGIC_VECTOR(0 DOWNTO 0);
    addrb : IN STD_LOGIC_VECTOR(8 DOWNTO 0);
    dinb : IN STD_LOGIC_VECTOR(31 DOWNTO 0);
    doutb : OUT STD_LOGIC_VECTOR(31 DOWNTO 0)
  );
END COMPONENT;

COMPONENT timing_gen_dpbram_1024x32
  PORT (
    clka : IN STD_LOGIC;
    wea : IN STD_LOGIC_VECTOR(0 DOWNTO 0);
    addra : IN STD_LOGIC_VECTOR(9 DOWNTO 0);
    dina : IN STD_LOGIC_VECTOR(31 DOWNTO 0);
    douta : OUT STD_LOGIC_VECTOR(31 DOWNTO 0);
    clkb : IN STD_LOGIC;
    enb : IN STD_LOGIC;
    web : IN STD_LOGIC_VECTOR(0 DOWNTO 0);
    addrb : IN STD_LOGIC_VECTOR(9 DOWNTO 0);
    dinb : IN STD_LOGIC_VECTOR(31 DOWNTO 0);
    doutb : OUT STD_LOGIC_VECTOR(31 DOWNTO 0)
  );
END COMPONENT;

--
-- Signal declarations
--

signal addra : std_logic_vector(BRAM_DEPTH_TWO_TO_N-1 downto 0) := (others => '0');
signal addrb : std_logic_vector(BRAM_DEPTH_TWO_TO_N-1 downto 0) := (others => '0');
signal clka  : std_logic := '0';
signal clkb  : std_logic := '0';
signal dina  : std_logic_vector(31 downto 0) := (others => '0');  
signal douta : std_logic_vector(31 downto 0) := (others => '0');
signal doutb : std_logic_vector(31 downto 0) := (others => '0');  
signal enb   : std_logic := '0';
signal wea   : std_logic_vector(0 downto 0) := (others => '0');

begin -- architecture: RTL of Timing Generator

timing_gen_core : timing_gen_ext
   generic map
   (
      BRAM_DEPTH_TWO_TO_N => BRAM_DEPTH_TWO_TO_N
   )                                                 
   port map(
      i_emif_clk    => i_emif_clk,           
      i_ABus        => i_ABus,               
      i_DBus        => i_DBus, 
      o_DBus        => o_DBus, 
      i_wr_en       => i_wr_en,
      i_rd_en       => i_rd_en,
      i_cs          => i_cs,   
      o_irq         => o_irq,  
      i_ilevel      => i_ilevel,
      i_ivector     => i_ivector,
		o_clka        => clka,  
		o_addra       => addra, 
		o_dina        => dina,  
		i_douta       => douta, 
		o_wea         => wea(0),   
      o_clkb        => clkb,  
		o_addrb       => addrb, 
		i_doutb       => doutb, 
		o_enb         => enb,   
      i_clk         => i_clk,   
      i_clk_en      => i_clk_en,
		i_wait_inp    => i_wait_inp,
      o_output      => o_output
      
      );

BRAM_512: if BRAM_DEPTH_TWO_TO_N = 9 generate
opcode_bram : timing_gen_dpbram_512x32
	PORT MAP (
		clka => clka,
		wea => wea,
		addra => addra,
		dina => dina,
		douta => douta,
		clkb => clkb,
		enb => enb,
		web => (others => '0'),
		addrb => addrb,
		dinb => (others => '0'),
		doutb => doutb
	);
end generate BRAM_512;

BRAM_1024: if BRAM_DEPTH_TWO_TO_N = 10 generate
opcode_bram : timing_gen_dpbram_1024x32
	PORT MAP (
		clka => clka,
		wea => wea,
		addra => addra,
		dina => dina,
		douta => douta,
		clkb => clkb,
		enb => enb,
		web => (others => '0'),
		addrb => addrb,
		dinb => (others => '0'),
		doutb => doutb
	);    
end generate BRAM_1024;

end rtl;