--- Title: base_module.vhd
--- Description: 
---
---     o  0
---     | /       Copyright (c) 2010
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 09/23/2010
--- Version: 1.01
--- Revisions: 
---   1.00 - Initial release.
---   1.01 - Changed nomenclature of IRQ assignments (to CPUs, instead of ARM or DSP).
---          Added back DCM reset logic, with a generic to disable it.
---          Added Device DNA read logic, with a generic to disable it.

library IEEE;
library UNISIM;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use UNISIM.VCOMPONENTS.ALL;

library WORK;
use WORK.MityDSP_L138_pkg.ALL;

entity base_module is
   generic (
      CONFIG       : string := "UNKNOWN"; -- "MityDSP_L138", "MityARM_1808"
      IRQ0_CPU     : integer range 0 to 1 := 0; -- IRQ0 CPU enumeration.  On L138, 0 is ARM and 1 is DSP.
      IRQ1_CPU     : integer range 0 to 1 := 1; -- IRQ1 CPU enumeration.  On L138, 0 is ARM and 1 is DSP.
      GEN_DCM_RST  : boolean := FALSE; -- Enables generation of DCM reset logic.  Uses STARTUP_SPARTAN6 primitive.
      GEN_DNA_PORT : boolean := FALSE  -- Enables generation of Device DNA read logic.  Uses DNA_PORT primitive.
   );
   port (
      ema_clk         : in  std_logic;
      i_cs            : in  std_logic;
      i_ID            : in  std_logic_vector(7 downto 0);    -- assigned Application ID number, 0xFF if unassigned
      i_version_major : in  std_logic_vector(3 downto 0);    -- major version number 1-15
      i_version_minor : in  std_logic_vector(3 downto 0);    -- minor version number 0-15
      i_year          : in  std_logic_vector(4 downto 0);    -- year since 2000
      i_month         : in  std_logic_vector(3 downto 0);    -- month (1-12)
      i_day           : in  std_logic_vector(4 downto 0);    -- day (1-32)
      i_ABus          : in  std_logic_vector(5 downto 0);
      i_DBus          : in  std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in  std_logic;
      i_rd_en         : in  std_logic;
      i_be_r          : in  std_logic_vector(1 downto 0);
      
      i_irq_map       : in  bus16_vector(1 downto 0) := (others=>(others=>'0'));
      o_irq_output    : out std_logic_vector(1 downto 0);
      
      i_dcm_status    : in  std_logic_vector(2 downto 0);  -- hook to EMIF DCM status lines
      i_dcm_lock      : in  std_logic;                     -- hook to EMIF DCM lock line
      o_dcm_reset     : out std_logic                      -- hook to EMIF DCM reset line
   );
end base_module;

architecture rtl of base_module is

constant CORE_APPLICATION_ID: std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR( 0, 8);
constant CORE_VERSION_MAJOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 1, 4);
constant CORE_VERSION_MINOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 1, 4);
constant CORE_YEAR:           std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR(10, 5);
constant CORE_MONTH:          std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR(09, 4);
constant CORE_DAY:            std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR(27, 5);

constant OFFSET_BASE_VERSION: std_logic_vector(7 downto 0) := x"00";
constant OFFSET_IRQ0_MASKED:  std_logic_vector(7 downto 0) := x"02";
constant OFFSET_IRQ0_ENABLES: std_logic_vector(7 downto 0) := x"04";
constant OFFSET_IRQ1_MASKED:  std_logic_vector(7 downto 0) := x"06";
constant OFFSET_IRQ1_ENABLES: std_logic_vector(7 downto 0) := x"08";
constant OFFSET_IRQ_CPU_MAP:  std_logic_vector(7 downto 0) := x"0A";
constant OFFSET_FPGA_VERSION: std_logic_vector(7 downto 0) := x"0C";
constant OFFSET_DEVDNA_15_00: std_logic_vector(7 downto 0) := x"10";
constant OFFSET_DEVDNA_31_16: std_logic_vector(7 downto 0) := x"12";
constant OFFSET_DEVDNA_47_32: std_logic_vector(7 downto 0) := x"14";
constant OFFSET_DEVDNA_56_48: std_logic_vector(7 downto 0) := x"16";

signal ABus : std_logic_vector(7 downto 0);

signal fpga_version_reg : std_logic_vector(15 downto 0) := (others=>'0');
signal base_version_reg : std_logic_vector(15 downto 0) := (others=>'0');

signal fpga_version_rd : std_logic;
signal base_version_rd : std_logic;

signal scratch_ram_l : bus8_vector(0 to 31) := (others=>(others=>'0'));
signal scratch_ram_u : bus8_vector(0 to 31) := (others=>(others=>'0'));
attribute ram_style : string;
attribute ram_style of scratch_ram_l: signal is "distributed";
attribute ram_style of scratch_ram_u: signal is "distributed";

signal masked_irqs : bus16_vector(1 downto 0);
signal irq_enables : bus16_vector(1 downto 0) := (others=>(others=>'0'));

signal osc_clk : std_logic; -- ~50MHz internal oscillator clock (in STARTUP_SPARTAN6 primitive)
constant clken4ms_cntr_TC : integer := 200000; -- With ~50MHz osc_clk, 200000 counts = ~4ms
signal clken4ms_cntr : std_logic_vector(17 downto 0) := (others=>'0');
signal clken4ms : std_logic := '0';

signal dcm_reset_shift : std_logic_vector(15 downto 0) := "0111111111111111";
signal dcm_reset : std_logic := '0';
signal dcm_lock_r1 : std_logic := '0';
signal dcm_lock_r2 : std_logic := '0';

constant devdna_clk_cntr_TC : integer := 25; -- 100MHz / (25 * 2) = 2MHz
signal devdna_clk_cntr : std_logic_vector(4 downto 0) := (0=>'1',others=>'0');
signal devdna_clken : std_logic := '0';
constant devdna_cntr_TC : integer := 57;
signal devdna_cntr : std_logic_vector(5 downto 0) := "000000";
signal devdna_dout : std_logic;
signal devdna_clk : std_logic := '0';
signal devdna_read : std_logic := '1';
signal devdna_shift : std_logic := '0';
signal devdna_reg : std_logic_vector(63 downto 0) := (others=>'0');

begin -- architecture: rtl

assert CONFIG="MityDSP_L138"
report "CONFIG generic must be MityDSP_L138."
severity FAILURE;

fpga_version : core_version
   port map(
      clk           => ema_clk,
      rd            => fpga_version_rd,
      ID            => i_ID,
      version_major => i_version_major,
      version_minor => i_version_minor,
      year          => i_year,
      month         => i_month,
      day           => i_day,
      o_data        => fpga_version_reg
      );

base_version : core_version
   port map(
      clk           => ema_clk,
      rd            => base_version_rd,
      ID            => CORE_APPLICATION_ID,
      version_major => CORE_VERSION_MAJOR,
      version_minor => CORE_VERSION_MINOR,
      year          => CORE_YEAR,
      month         => CORE_MONTH,
      day           => CORE_DAY,
      o_data        => base_version_reg
      );

ABus <= '0' & i_ABus & '0';

reg_read : process (ema_clk)
begin
   if rising_edge(ema_clk) then
      base_version_rd <= '0';
      fpga_version_rd <= '0';
      o_DBus <= x"0000"; -- default
      
      if i_cs = '1' then
         case ABus is
            when OFFSET_BASE_VERSION =>
               o_DBus <= base_version_reg;
               base_version_rd <= i_rd_en;
            when OFFSET_IRQ0_MASKED =>
               o_DBus <= masked_irqs(0);
            when OFFSET_IRQ0_ENABLES =>
               o_DBus <= irq_enables(0);
            when OFFSET_IRQ1_MASKED =>
               o_DBus <= masked_irqs(1);
            when OFFSET_IRQ1_ENABLES =>
               o_DBus <= irq_enables(1);
            when OFFSET_IRQ_CPU_MAP =>
               o_DBus(0) <= conv_std_logic_vector(IRQ0_CPU,1)(0);
               o_DBus(1) <= conv_std_logic_vector(IRQ1_CPU,1)(0);
            when OFFSET_FPGA_VERSION =>
               o_DBus <= fpga_version_reg;
               fpga_version_rd <= i_rd_en;
            when OFFSET_DEVDNA_15_00 =>
               o_DBus <= devdna_reg(15 downto 0);
            when OFFSET_DEVDNA_31_16 =>
               o_DBus <= devdna_reg(31 downto 16);
            when OFFSET_DEVDNA_47_32 =>
               o_DBus <= devdna_reg(47 downto 32);
            when OFFSET_DEVDNA_56_48 =>
               o_DBus <= devdna_reg(63 downto 48);
            when others =>
               if i_ABus(5) = '1' then
                  o_DBus <= scratch_ram_u(conv_integer(i_ABus(4 downto 0)))
                          & scratch_ram_l(conv_integer(i_ABus(4 downto 0)));
               else
                  o_DBus <= x"0000";
               end if;
         end case;
      end if;
   end if;
end process reg_read;

reg_write : process (ema_clk)
begin
   if rising_edge(ema_clk) then
      if i_cs='1' and i_wr_en='1' then
         if i_ABus(5)='1' then
             if i_be_r(1)='1' then
                 scratch_ram_u(conv_integer(i_ABus(4 downto 0))) <= i_DBus(15 downto 8);
             end if;
             if i_be_r(0)='1' then
                 scratch_ram_l(conv_integer(i_ABus(4 downto 0))) <= i_DBus(7 downto 0);
             end if;
         end if;
         
         case ABus is
            when OFFSET_IRQ0_ENABLES =>
              irq_enables(0) <= i_DBus;
            when OFFSET_IRQ1_ENABLES =>
              irq_enables(1) <= i_DBus;
            when others => NULL;
         end case;
      end if;
   end if;
end process reg_write;

-- IRQ masking logic
gen_irqs : for i in 0 to 1 generate
begin
   o_irq_output(i) <= '1' when masked_irqs(i) /= x"0000" else '0';

   gen_mask: for j in 0 to 15 generate
   begin
      masked_irqs(i)(j) <= i_irq_map(i)(j) and irq_enables(i)(j);
   end generate gen_mask;
end generate gen_irqs;

gen_dcm_rst_logic : if GEN_DCM_RST = TRUE generate
begin
   s6_startup : STARTUP_SPARTAN6
      port map (
         CFGCLK    => open,    -- 1-bit Configuration logic main clock output.
         CFGMCLK   => osc_clk, -- 1-bit Configuration internal oscillator clock output.
         EOS       => open,    -- 1-bit Active high output signal indicates the End Of Configuration.
         CLK       => '0',     -- 1-bit User startup-clock input
         GSR       => '0',     -- 1-bit Global Set/Reset input (GSR cannot be used for the port name)
         GTS       => '0',     -- 1-bit Global 3-state input (GTS cannot be used for the port name)
         KEYCLEARB => '0'      -- 1-bit Clear AES Decrypter Key input from Battery-Backed RAM (BBRAM)
      );
   
   dcm_reset_trigger : process(osc_clk)
   begin
      if rising_edge(osc_clk) then
         if clken4ms_cntr = conv_std_logic_vector(clken4ms_cntr_TC, 18) then
            clken4ms_cntr <= (others=>'0');
            clken4ms <= '1';
         else
            clken4ms_cntr <= clken4ms_cntr + '1';
            clken4ms <= '0';
         end if;
         
         dcm_lock_r1 <= i_dcm_lock;
         dcm_lock_r2 <= dcm_lock_r1;
         
         if (clken4ms='1' and dcm_lock_r2='0') or 
            (dcm_lock_r1='0' and dcm_lock_r2='1') or dcm_reset_shift(15)='1' then
            dcm_reset_shift <= dcm_reset_shift(14 downto 0) & dcm_reset_shift(15);
         end if;
      end if;    
   end process dcm_reset_trigger;
   
   dcm_reset <= i_dcm_status(1) or dcm_reset_shift(15);
end generate gen_dcm_rst_logic;

o_dcm_reset <= dcm_reset;

gen_devdna_logic : if GEN_DNA_PORT = TRUE generate
begin
   devdna_inst : DNA_PORT
      generic map (
         SIM_DNA_VALUE => X"0123456789ABCDEF"  -- Specifies the Pre-programmed factory ID value
      )
      port map (
         DOUT  => devdna_dout,  -- 1-bit DNA output data
         CLK   => devdna_clk,   -- 1-bit Clock input
         DIN   => '0',          -- 1-bit User data input pin
         READ  => devdna_read,  -- 1-bit Active high load DNA, active low read input
         SHIFT => devdna_shift  -- 1-bit Active high shift enable input
      );

   devdna_proc : process (ema_clk)
   begin
      if rising_edge(ema_clk) then
         if devdna_clk_cntr = conv_std_logic_vector(devdna_clk_cntr_TC, 5) then
            devdna_clk_cntr <= (0=>'1',others=>'0');
            devdna_clken <= '1';
         else
            devdna_clk_cntr <= devdna_clk_cntr + '1';
            devdna_clken <= '0';
         end if;
         
         if devdna_clken = '1' then
            if devdna_cntr < conv_std_logic_vector(devdna_cntr_TC, 6) then
               devdna_clk <= not devdna_clk;
               
               if devdna_clk = '1' then
                  devdna_cntr <= devdna_cntr + '1';
                  devdna_reg <= devdna_reg(62 downto 0) & devdna_dout;
                  devdna_read <= '0';
                  devdna_shift <= '1';
               end if;
            else
               devdna_read <= '0';
               devdna_shift <= '0';
            end if;
         end if;
      end if;
   end process devdna_proc;
end generate gen_devdna_logic;

end rtl;
