--- Title: spit.vhd
--- Description: Implementation of SPI (Serial Peripheral Interface) Core 
---
---     o  0
---     | /       Copyright (c) 2007-2010
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 01/03/2012
--- Version: 2.01
--- Revisions: 
---    1.00  Adapted for MityDSP-L138 family
---    1.01  Make compliant with standard CPOL / CPHA specification
---    1.02  Add optional port signal WORD_DELAY for per-core word spacing
---    1.03  Don't allow chip sync line to go high between FIFO sets
---    2.00  Rewrite.  Support CPOL and CHPA properly.  Get rid of word delay.
---          Support clock division in the core and use only EMIFA_CLOCK for now
---    2.01  Support Chip Select Toggle between word streams (Micro-WIU feature)
---    2.02  Support chip select toggle for CPHA=1 correctly, support leaving CS asserted
---          during word delay operations.

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
library UNISIM;
use UNISIM.VCOMPONENTS.ALL;
use work.MityDSP_L138_pkg.ALL;

entity spi is
   Port 
   (  
      emif_clk   : in  std_logic;
      i_ABus   : in std_logic_vector(5 downto 0);
      i_DBus   : in std_logic_vector(15 downto 0);
      o_DBus   :out std_logic_vector(15 downto 0);
      i_wr_en  : in std_logic;
      i_rd_en  : in std_logic;
      i_cs     : in std_logic;
      o_irq    :out std_logic := '0';
      i_ilevel       : in std_logic_vector(1 downto 0) := "00";      
      i_ivector      : in std_logic_vector(3 downto 0) := "0000";
      -- SPI interface signals		
      o_sclk     : out std_logic;                      -- SPI output clock
      o_cs_n     : out std_logic_vector(7 downto 0) := "11111111"; -- SYNC output 
      o_mosi     : out std_logic;                      -- data output 
      i_miso     : in  std_logic;                      -- data input 
		
      -- In/Out FIFO interfaces (NO FIRST WORD FALL THROUGH)
      --    Synchronous on emif_clock
      i_fifo_depth      : in std_logic_vector(2 downto 0) := "110";  -- DEPTH=2^(CONV_INTEGER(i_fifo_depth)+4), MAX 2048
      o_mosi_fifo_wr    : out std_logic := '0';
      o_mosi_fifo_rd    : out std_logic := '0';
      o_mosi_fifo_in    : out std_logic_vector(31 downto 0) := (others=>'0');  -- fifo data input
      i_mosi_fifo_out   : in std_logic_vector(31 downto 0) := (others=>'0');  -- fifo data output
      i_mosi_write_cnt  : in std_logic_vector(11 downto 0) := (others=>'0');
      i_mosi_empty      : in std_logic := '0';

      o_miso_fifo_wr    : out std_logic := '0';
      o_miso_fifo_rd    : out std_logic := '0';
      o_miso_fifo_in    : out std_logic_vector(31 downto 0) := (others=>'0');  -- fifo data input
      i_miso_fifo_out   : in std_logic_vector(31 downto 0) := (others=>'0');  -- fifo data output
      i_miso_read_cnt   : in std_logic_vector(11 downto 0) := (others=>'0');
      i_miso_empty      : in std_logic := '0';
      
      o_fifo_rst        : out std_logic := '0'

    );
end spi;						 

architecture rtl of spi is

-- All Used components should be declared first.
--
constant CORE_VERSION_MAJOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 2, 4);
constant CORE_VERSION_MINOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 1, 4);
constant CORE_ID:             std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR( 14, 8);
constant CORE_YEAR:           std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 13, 5);
constant CORE_MONTH:          std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 11, 4);
constant CORE_DAY:            std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 14, 5);

signal version_reg       : std_logic_vector(15 downto 0) := (others=>'0');  -- version register
signal ver_rd            : std_logic := '0';

signal miso : std_logic := '0';

-- interrupt status and enable register fields
signal mosi_hf    : std_logic := '0';              -- FIFO quarter half full flag
signal mosi_tc    : std_logic := '0';              -- SPI transfer complete flag
signal miso_hf    : std_logic := '0';

signal mosi_hfie  : std_logic := '0';              -- FIFO quarter half full interrupt enable
signal mosi_tcie  : std_logic := '0';              -- SPI transfer complete interrupt enable
signal miso_hfie  : std_logic := '0';              -- FIFO quarter half full interrupt enable
signal miso_daie  : std_logic := '0';              -- data available interrupt enable

-- FIFO control and data lines
signal mosi_fifo_wr  : std_logic := '0';
signal mosi_fifo_rd  : std_logic := '0';
signal mosi_fifo_in  : std_logic_vector(31 downto 0) := (others=>'0');  -- fifo data input
signal mosi_fifo_empty_r1 : std_logic := '1';
signal data_avail : std_logic := '0';

signal miso_fifo_wr  : std_logic := '0';
signal miso_fifo_rd  : std_logic := '0';
signal miso_fifo_in  : std_logic_vector(31 downto 0) := (others=>'0');  -- fifo data input

signal dwidth    : std_logic_vector(2 downto 0);
signal rcv_en    : std_logic := '1';
signal sclk_ctrl : std_logic := '1';
signal cpol      : std_logic := '0';
signal cpha      : std_logic := '0';
signal clkout    : std_logic := '0';

signal sclk, sclk_r1 : std_logic := '0';
signal shift_cnt     : std_logic_vector(5 downto 0) := (others=>'1');
signal DATA_WIDTH    : integer range 0 to 32 := 8;

signal fifo_data_avail : std_logic := '0';
signal mosi_shift_reg  : std_logic_vector(31 downto 0);

signal loopback        : std_logic := '0';
signal half_bit        : integer range 0 to 11 := 5;

signal cs_en : std_logic_vector(2 downto 0) := "001";
signal cs : std_logic_vector(7 downto 0) := (others=>'1');
signal spi_div : std_logic_vector(11 downto 0) := CONV_STD_LOGIC_VECTOR(50,12); 
signal spi_clk_cnt : std_logic_vector(11 downto 0) := (others=>'0');
signal go, go_tog, go_tog_r1 : std_logic := '0';
signal fifo_mask : std_logic_vector(31 downto 0) := x"FFFFFFFF";
signal fifo_rst : std_logic := '0';

signal word_delay : std_logic_vector(14 downto 0) := (others=>'0');
signal word_delay_en : std_logic := '0';
signal delay_cnt : std_logic_vector(14 downto 0) := (others=>'0');
signal word_delay_csmask : std_logic := '0';

-- for debug remove when done
signal word_count : std_logic_vector(7 downto 0) := x"00";

type SPISTATES is (IDLE, ASSERT_CS, SHIFTDATA, DEASSERT_CS, STALL);
signal smstate : SPISTATES := IDLE;

begin

version : core_version
   port map(
      clk           => emif_clk,                  -- system clock
      rd            => ver_rd,               -- read enable
      ID            => CORE_ID,              -- assigned ID number, 0xFF if unassigned
      version_major => CORE_VERSION_MAJOR,   -- major version number 1-15
      version_minor => CORE_VERSION_MINOR,   -- minor version number 0-15
      year          => CORE_YEAR,            -- year since 2000
      month         => CORE_MONTH,           -- month (1-12)
      day           => CORE_DAY,             -- day (1-31)
      ilevel        => i_ilevel,
      ivector       => i_ivector,
      o_data        => version_reg
      );

miso <= i_miso when loopback='0' else mosi_shift_reg(DATA_WIDTH-1);
half_bit <= CONV_INTEGER(i_fifo_depth)+3;

mosi_hf  <= not i_mosi_write_cnt(half_bit);
miso_hf  <= i_miso_read_cnt(half_bit);

o_irq <= (mosi_hf  and mosi_hfie) or
         (mosi_tc  and mosi_tcie) or
         (miso_hf  and miso_hfie) or
         (data_avail and miso_daie);       

o_sclk <= clkout when sclk_ctrl='1' else (sclk_r1 xor cpol);
o_mosi <= mosi_shift_reg(DATA_WIDTH-1);
o_cs_n <= cs;
o_mosi_fifo_wr    <= mosi_fifo_wr;
o_mosi_fifo_rd    <= mosi_fifo_rd;
o_mosi_fifo_in    <= mosi_fifo_in;
o_miso_fifo_wr    <= miso_fifo_wr;
o_miso_fifo_rd    <= miso_fifo_rd;
o_fifo_rst        <= fifo_rst;

fifo_in_mask : for i in 0 to 31 generate
begin
o_miso_fifo_in(i)  <= fifo_mask(i) and miso_fifo_in(i);
end generate fifo_in_mask;

--* Handle read requests from the processor
--/
read_mux_regs : process (emif_clk) is
begin
	if emif_clk'event and emif_clk='1' then

      ver_rd <= '0';
      miso_fifo_rd <= '0';
      
      if i_miso_empty='0' and data_avail='0' then
          data_avail <= '1';
          miso_fifo_rd <= '1';
      elsif i_cs='1' and i_ABus="000101" and i_rd_en='1' then
          data_avail <= not i_miso_empty;
          miso_fifo_rd <= '1';
      end if;
      
      -- address decoding
      if i_cs = '0' then
         o_DBus <= (others=>'0');
      else
         case i_ABus is
            when "000000" =>   
               o_DBus <=  version_reg;
               ver_rd <= i_rd_en;
            when "000001" =>	
   			   o_DBus <= x"0" & "0" & word_delay_csmask & "00" & loopback & dwidth & cpol & cpha & sclk_ctrl & rcv_en;             
   			when "000011" =>	
   			   o_DBus <= '0' & i_fifo_depth & i_mosi_write_cnt; 
            when "000100" =>	
   			   o_DBus <= i_miso_fifo_out(15 downto 0); 
            when "000101" =>	
   			   o_DBus <= i_miso_fifo_out(31 downto 16);
   			when "000110" =>	
   			   o_DBus <= not data_avail & i_fifo_depth & i_miso_read_cnt; 
   			when "000111" =>	
   			   o_DBus <= x"00" & '0' & miso_hfie & '0' & miso_daie & '0' & mosi_hfie & '0' & mosi_tcie; 
   			when "001000" =>	
   			   o_DBus <= x"00" & '0' & miso_hf & '0' & data_avail & '0' & mosi_hf & '0' & mosi_tc;
   			when "001001" =>
   			   o_DBus <= word_count(7) & cs_en & spi_div; 
            when "001010" =>
               O_DBus <= word_delay_en & word_delay;
   			when others =>	
               o_DBus <= (others=>'0');
         end case;
      end if;
   end if;
end process read_mux_regs;


--* Decode register write requests.
--/
wr_ctl_reg : process(emif_clk)
begin
	if emif_clk='1' and emif_clk'event then
      
		mosi_fifo_wr <= '0';
		fifo_rst <= '0';
		
		if i_cs='1' and i_wr_en='1' and i_ABus="001000" and i_DBus(0)='1' then
			mosi_tc <= '0';
		elsif smstate = DEASSERT_CS and sclk_r1='0' and sclk='1' then
			mosi_tc <= '1';
        end if;
        
		if i_cs = '1' and i_wr_en = '1' then
			case i_ABus is
			   when "000001" => 
			      rcv_en         <= i_DBus(0);
			      sclk_ctrl      <= i_DBus(1);
			      cpha           <= i_DBus(2);
			      cpol           <= i_DBus(3);
			      dwidth         <= i_DBus(6 downto 4); 
  			      loopback       <= i_DBus(7);
				  if i_DBus(8)='1' and smstate=IDLE then
					    go_tog <= not go_tog;
				  end if;
				  if i_DBus(9)='1' and smstate=IDLE then
				        fifo_rst <= '1';
				  end if;
  			      word_delay_csmask <= i_DBus(10);
    			when "000100" =>  
    			   mosi_fifo_in(15 downto 0) <= i_DBus;
    			when "000101" =>  
    			   mosi_fifo_in(31 downto 16) <= i_DBus;
    			   mosi_fifo_wr <= '1';
    			when "000111" =>
    			   mosi_tcie  <= i_DBus(0);
    			   mosi_hfie  <= i_DBus(2);
    			   miso_daie  <= i_DBus(4);
    			   miso_hfie  <= i_DBus(6);
  		        when "001001" =>
   			       spi_div      <= i_DBus(11 downto 0); 
   			       cs_en        <= i_DBus(14 downto 12);
                when "001010" =>
                   word_delay_en <= i_DBus(15);
                   word_delay    <= i_DBus(14 downto 0);
    			when others => NULL;
			end case;
	   end if;
	    
	   case dwidth is
	      when "000"  => 
	          DATA_WIDTH <= 4;
	          fifo_mask <= x"0000000F";
	      when "001"  => 
	          DATA_WIDTH <= 8;
	          fifo_mask <= x"000000FF";
	      when "010"  => 
	          DATA_WIDTH <= 12;
	          fifo_mask <= x"00000FFF";
	      when "011"  => 
	          DATA_WIDTH <= 16;
	          fifo_mask <= x"0000FFFF";
	      when "100"  => 
	          DATA_WIDTH <= 20;
	          fifo_mask <= x"000FFFFF";
	      when "101"  => 
	          DATA_WIDTH <= 24;
	          fifo_mask <= x"00FFFFFF";
	      when "110"  => 
	          DATA_WIDTH <= 28;
       	          fifo_mask <= x"0FFFFFFF";
	      when "111"  => 
	          DATA_WIDTH <= 32;
       	          fifo_mask <= x"FFFFFFFF";
	      when others => 
	          DATA_WIDTH <= 32;
       	          fifo_mask <= x"FFFFFFFF";
	   end case;	
	          
	end if;
end process wr_ctl_reg;

--* SPI clock generation
--/
spi_clk_gen : process(emif_clk)
begin
    if rising_edge(emif_clk) then
        if spi_clk_cnt >= spi_div then
            spi_clk_cnt <= CONV_STD_LOGIC_VECTOR(1,12);
            sclk <= not sclk;
        else
            spi_clk_cnt <= spi_clk_cnt+'1';
        end if;
    end if;
end process spi_clk_gen;

--* Serial Shifting Process using 7-bit counter 
--/
spi_proc : process(emif_clk)
begin

    if rising_edge(emif_clk) then
        sclk_r1 <= sclk;
        mosi_fifo_rd <= '0';
        miso_fifo_wr <= '0';        
	    go_tog_r1 <= go_tog;
		
		case smstate is
		    when IDLE =>
			clkout <= cpol;
		        if go_tog_r1 /= go_tog then
			    mosi_fifo_rd <= '1';
		            cs <= (others=>'1');
		            shift_cnt <= (others=>'0');
		            miso_fifo_in <= (others=>'0');
		            word_count <= x"01";
		        end if;
		        -- need an extra clock here...
		        if mosi_fifo_rd='1' then
		            smstate <= ASSERT_CS;
		        end if;
		        
		    when ASSERT_CS =>
				if sclk_r1='1' and sclk='0' then
					mosi_shift_reg <= i_mosi_fifo_out;
					mosi_fifo_empty_r1 <= i_mosi_empty;
					cs(CONV_INTEGER(cs_en)) <= '0';
					smstate <= SHIFTDATA;
				end if;
				
		    -- Break into two states based on CPHA?, not sure if it's 
		    -- any more or less efficient
		    when SHIFTDATA =>
				clkout <= sclk xor cpol;
				if (cpha='0' and sclk_r1='0' and sclk='1') or
				   (cpha='1' and sclk_r1='1' and sclk='0') then
				   miso_fifo_in  <= miso_fifo_in(30 downto 0) & miso;
				   shift_cnt <= shift_cnt+'1';
				elsif (cpha='0' and sclk_r1='1' and sclk='0') or
				      (cpha='1' and sclk_r1='0' and sclk='1') then
				   if (cpha='1' and shift_cnt /= CONV_STD_LOGIC_VECTOR(0,6)) or
				      (cpha='0') then
				       mosi_shift_reg <= mosi_shift_reg(30 downto 0) & '0';
				   end if;
				   if shift_cnt = CONV_STD_LOGIC_VECTOR(1,6) then
				       mosi_fifo_rd <= '1';
				       word_count <= word_count+'1';
				   end if;
				   if shift_cnt = CONV_STD_LOGIC_VECTOR(DATA_WIDTH,6) then
					   shift_cnt <= (others=>'0');
					   miso_fifo_wr <= rcv_en;
					   if mosi_fifo_empty_r1='1' then
							smstate <= DEASSERT_CS;
							clkout <= cpol;
					   else
							if word_delay_en = '1' then
								smstate <= STALL;
								delay_cnt <= word_delay;
								clkout <= cpol;
							end if;
							mosi_shift_reg <= i_mosi_fifo_out;
							mosi_fifo_empty_r1 <= i_mosi_empty;
							shift_cnt <= CONV_STD_LOGIC_VECTOR(0,6);
					   end if;
				   end if;
				end if;

			-- some devices require deassertion of chip select for each word transferred.  The CPU can do this,
			-- but is is terribly inefficient to start a new transfer everyword.  Use state logic to delay for us...
			when STALL =>
				if sclk_r1='0' and sclk='1' then
				    if word_delay_csmask='0' then
						cs <= (others=>'1');
					end if;
					delay_cnt <= delay_cnt-'1';
				end if;
				if delay_cnt = CONV_STD_LOGIC_VECTOR(0,15) then
					smstate <= ASSERT_CS;
				end if;

		    when DEASSERT_CS =>
		        if sclk_r1='0' and sclk='1' then
		            cs <= (others=>'1');
		            smstate <= IDLE;
		        end if;
		        
		    when others =>
	            smstate <= IDLE;
		end case;
    end if;
end process spi_proc;      

end rtl;
