--- Title: SPI_iface.vhd
--- Description: 
---
--- SPI Bus interface for FPGA with OMAP-L138 Core Architecture.
---
---     o  0
---     | /       Copyright (c) 2010
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 1/03/2012
--- Version: 1.00
--- Revisions:
---   1.00 - Baseline

library WORK;
library IEEE;
library UNISIM;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use UNISIM.VCOMPONENTS.ALL;
use WORK.MityDSP_L138_pkg.ALL;

-- REQUIREMENTS
-- i_core_clk >= 4*i_spi_clk
entity SPI_iface is
   generic ( 
      DECODE_BITS   : integer range 1 to 9 := 5;
      CONFIG        : string := "UNKNOWN"        -- "MityDSP_L138" 
   );
   port (      
      -- SPI direct-connect signals
      i_spi_cs_n    : in  std_logic;
      i_spi_clk     : in  std_logic;
      i_spi_din     : in  std_logic;
      o_spi_dout    : out std_logic;
      
      -- FPGA core interface signals
      i_core_clk    : in  std_logic;
      o_core_be     : out std_logic_vector(1 downto 0);
      o_core_addr   : out std_logic_vector(5 downto 0);
      o_core_cs     : out std_logic_vector((2**DECODE_BITS)-1 downto 0);
      o_core_edi    : out std_logic_vector(15 downto 0);
      i_core_edo    : in  bus16_vector((2**DECODE_BITS)-1 downto 0);
      o_core_rd     : out std_logic;
      o_core_wr     : out std_logic
   );
end SPI_iface;

architecture rtl of SPI_iface is

signal shift_cnt : std_logic_vector(4 downto 0) := (others=>'0');
signal word_addr, word_addr_out : std_logic_vector(14 downto 0) := (others=>'0');
signal cs : std_logic_vector((2**DECODE_BITS)-1 downto 0);
signal cmd_read : std_logic := '1';
signal data_in : std_logic_vector(15 downto 0) := (others=>'0');
signal pulse_read_tog, pulse_read_tog_m, pulse_read_tog_r1, pulse_read_tog_r2 : std_logic := '0';
signal pulse_write_tog : std_logic := '0';
signal pulse_write_r : std_logic_vector(9 downto 0) := (others=>'0');
signal bus_out : std_logic_vector(15 downto 0) := (others=>'0');
signal data_out : std_logic_vector(15 downto 0) := (others=>'0');

type SPISTATES is (READ_ADDR, READ_COMMAND, SHIFT_DATA);
signal spi_state : SPISTATES := READ_ADDR;

begin

-- SPI interface
--
spi_sm : process(i_spi_clk)
begin
    if i_spi_cs_n='1' then
       spi_state <= READ_ADDR;
       shift_cnt <= (others=>'0');
    else
       if rising_edge(i_spi_clk) then           
           case spi_state is
           
           -- read first 15 bits (address)
           when READ_ADDR    => 
               shift_cnt <= shift_cnt+'1';
               word_addr <= word_addr(13 downto 0) & i_spi_din;
               
               -- if first 15 bits are shifted in (address)
               if shift_cnt = CONV_STD_LOGIC_VECTOR(14, 5) then
			       spi_state <= READ_COMMAND;
	               word_addr_out <= word_addr(13 downto 0) & i_spi_din;
	               -- determine chip select from address
			       for count in 0 to (2**DECODE_BITS)-1 loop
			          if word_addr(DECODE_BITS+4 downto 5) = conv_std_logic_vector(count, DECODE_BITS) then
			             cs(count) <= '1';
			          else
			             cs(count) <= '0';
			          end if;
			       end loop;
               end if;
               
           when READ_COMMAND =>           
               cmd_read <= i_spi_din;
               spi_state <= SHIFT_DATA;
               shift_cnt <= (others=>'0');
               
           when SHIFT_DATA   =>
               shift_cnt <= shift_cnt+'1';
               data_in   <= data_in(14 downto 0) & i_spi_din;
               if shift_cnt = CONV_STD_LOGIC_VECTOR(15, 5) then
                   shift_cnt <= (others=>'0');
                   spi_state <= READ_ADDR;
	               if cmd_read='1' then
	                   pulse_read_tog <= not pulse_read_tog;
	               else
                       pulse_write_tog <= not pulse_write_tog;
                   end if;
               end if;
               
           when others => NULL;
           end case;
           
       end if;

       if falling_edge(i_spi_clk) then
           -- state machine updates on rising edge -- 1/2 clock timing constraint here
           -- (should be OK, this is a SPI bus....)
           case spi_state is
                      
           when SHIFT_DATA =>
               if shift_cnt = CONV_STD_LOGIC_VECTOR(0,5) then
	               data_out <= bus_out; -- clock crossing here, should be stable...
	           else
	           	   data_out <= data_out(14 downto 0) & '0';
	           end if;
           
           when others => NULL;
           
           end case;
       end if;
    end if;
end process spi_sm;

o_spi_dout <= data_out(15) when i_spi_cs_n='0' else 'Z';

-- clock crossing from SPI machine to EMIF machine
-- we should be able to TIG a lot of these if necessary for 
-- timing.
clock_cross : process(i_core_clk)
begin
    if rising_edge(i_core_clk) then  
        pulse_read_tog_m <= pulse_read_tog;      
        pulse_read_tog_r1 <= pulse_read_tog_m;      
        pulse_read_tog_r2 <= pulse_read_tog_r1;
        if pulse_read_tog_r1 /= pulse_read_tog_r2 then
            o_core_rd <= '1';
        else
            o_core_rd <= '0';
        end if;

        pulse_write_r <= pulse_write_r(8 downto 0) & pulse_write_tog;      
        if pulse_write_r(9) /= pulse_write_r(8) then
            o_core_wr <= '1';
        else
            o_core_wr <= '0';
        end if;

        o_core_cs   <= cs;
        o_core_addr <= word_addr_out(5 downto 0);
        o_core_edi  <= data_in;
        o_core_be   <= "11";
    end if;
end process clock_cross;

-- multiplex module data-out busses
merge_bus : process (i_core_clk)
   variable ored : std_logic_vector(15 downto 0);
begin
   if rising_edge(i_core_clk) then
      ored := x"0000";
      
      for count in 0 to (2**DECODE_BITS)-1 loop
         ored := ored or i_core_edo(count);
      end loop;
         
      bus_out <= ored;
   end if;
end process merge_bus;

end rtl;
