library IEEE;
library UNISIM;
library WORK;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use UNISIM.VCOMPONENTS.ALL;
use WORK.mitydsp_L138_pkg.all;

--*
--* Version history
--* 1.0 - Adapted for MityDSP-L138 family
--/

entity uart is
   Port ( 
      clk        : in  std_logic;
      i_ABus     : in std_logic_vector(5 downto 0);
      i_DBus     : in std_logic_vector(15 downto 0);
      o_DBus     : out std_logic_vector(15 downto 0);
      i_wr_en    : in std_logic;
      i_rd_en    : in std_logic;
      i_cs       : in std_logic;
      o_irq      : out std_logic := '0';
      i_ilevel   : in std_logic_vector(1 downto 0) := "00";      
      i_ivector  : in std_logic_vector(3 downto 0) := "0000";
      i_be       : in std_logic_vector(1 downto 0);

      -- UART LOGIC SIGNALS
      i_rcv          : in std_logic;
      o_xmit         : out std_logic;
      i_xmit_cts     : in std_logic := '0';   -- active low
      o_rcv_cts      : out std_logic;         -- active low
      i_dsr          : in std_logic := '0';   -- active low
      o_dtr          : out std_logic;         -- active low
      i_dcd          : in std_logic := '0';   -- active low
      i_ri           : in std_logic := '0';   -- active low
      o_xmit_enb     : out std_logic          -- transmit enable
   );
end uart;

--*
--* @short Register Transfer Logic Implementation of uart core entity.
--* 
--/
architecture rtl of uart is

-- All Used components should be declared first.
--
constant CORE_ID:             std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR( 3, 8);
constant CORE_VERSION_MAJOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 1, 4);
constant CORE_VERSION_MINOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 0, 4);
constant CORE_YEAR:           std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 10, 5);
constant CORE_MONTH:          std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 11, 4);
constant CORE_DAY:            std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 2, 5);

--*
--* Dual Port FIFO from Xilinx Core Generator.
--* Used for transmit and receive FIFO queues.
--*
--/
component uart_dfifo_64x8
	port (
   	clk        : IN std_logic;
   	sinit      : IN std_logic;
   	din        : IN std_logic_VECTOR(7 downto 0);
   	wr_en      : IN std_logic;
   	rd_en      : IN std_logic;
   	dout       : OUT std_logic_VECTOR(7 downto 0);
   	full       : OUT std_logic;
   	empty      : OUT std_logic;
   	data_count : OUT std_logic_VECTOR(6 downto 0));
end component;

component srl_counter is
   generic (
      NUM_STAGES : integer := 4
   );
   port (
      clk        :  in  std_logic;   -- input clock
      clk_en     :  out std_logic;    -- output clock enable (1 per cycle)
      DIVISIONS  :  bus4_vector(NUM_STAGES-1 downto 0) := (x"F",x"F",x"F",x"F")
      );
end component;

signal tbaud_counter : std_logic_vector(15 downto 0) := (others => '0'); -- transmit baud cnter modulus 2560
signal rbaud_counter : std_logic_vector(15 downto 0) := (others => '0'); -- receive baud cntr modulus 2560
signal rbaud_tc      : std_logic := '0';
signal rbaud_hc      : std_logic := '0';
signal tbaud_tc      : std_logic := '0';
signal rcv_sample    : std_logic := '0';
signal xmit_active   : std_logic := '0';              -- txa_empty, txa_full,
signal rcv_active    : std_logic := '0';              -- receive incoming
signal rcv_meta : std_logic := '1';
signal rcv_det       : std_logic := '0';              -- receive bit detected
signal rcv_det1      : std_logic := '0';              -- receive bit detected and latched
signal rcv_det2      : std_logic := '0';              -- double delay latch
signal xmit_rd_en    : std_logic := '0';              -- transmit FIFO read enable (UART side)
signal rcv_wr_en     : std_logic := '0';              -- receive write enable (UART side)
signal xmit_byte     : std_logic_vector(7 downto 0) := (others => '0');  -- active transmit byte
signal s_xmit_dout : std_logic_vector(7 downto 0) := (others => '0');  -- active transmit byte
signal rcv_reg       : std_logic_vector(7 downto 0) := (others => '0');  -- active recieve byte
signal tbitcnt       : std_logic_vector(3 downto 0) := (others => '0');  -- transmit bit count
signal rbitcnt       : std_logic_vector(3 downto 0) := (others => '0');  -- receive  bit count
signal tshift_reg    : std_logic_vector(7 downto 0) := (others => '0');  -- transmit shift register
signal rshift_reg    : std_logic_vector(7 downto 0) := (others => '0');  -- receive shift register
signal full_baud     : std_logic_vector(15 downto 0) := (others => '0'); -- applied baud counter
signal rx_fifo_count : std_logic_vector(6 downto 0) := (others => '0');  -- number of words in RX FIFO
signal tx_fifo_count : std_logic_vector(6 downto 0) := (others => '0');  -- number of words in TX FIFO

-- THIS STYLE IS OK for Register declaration...
--UART Registers
signal  RHR, --Receive Holding Reg
	THR --Transmit Holding Reg		
	: std_logic_vector(7 downto 0);	
signal IER : std_logic_vector(7 downto 0) := x"00"; --Interrupt Enable Reg     Bit 0 (RHR), Bit 1 (THR), Bit 2 (Modem)
signal MCR : std_logic_vector(7 downto 0) := x"00"; --Modem Control Reg        Bit 4 (loopback). Bit 0 DTR, Bit 1 RTS override
signal FCR : std_logic_vector(7 downto 0) := x"06"; --FIFO Control Reg         Bit 2 and Bit 1
signal DLL : std_logic_vector(7 downto 0) := x"B2"; --Divisor Latch LSB        Low Byte of BAUD Counter
signal DLH : std_logic_vector(7 downto 0) := x"01"; --Divisor Latch MSB        High Nibble of BAUD Counter
signal EFR : std_logic_vector(7 downto 0) := x"00"; --Enhanced Feature Reg     Bit 7 honor CTS, Bit 6 assert RTS
signal LCR : std_logic_vector(7 downto 0) := x"00"; --Line Control Reg         Bit 3 - parity enable, Bit 2 - stopbits (1 or 2), Bit [1:0] Parity (5,6,7,8)
signal MSR : std_logic_vector(7 downto 0) := x"00"; --Modem Status Reg         7,6,5,4 -> CD, RI, DSR, CTS (3-0 are changed since last read)
signal LSR : std_logic_vector(7 downto 0) := x"00"; --Line Status Reg          Bit 6,5,2,1
		               	
signal txa             : std_logic := '0';             -- 
signal r_txa           : std_logic := '0';             --
signal txa_we          : std_logic := '0';             --
signal r_rd_en         : std_logic;                    -- registered read enable
signal txa_full        : std_logic;                    -- async transmit fifo full
signal txa_empty       : std_logic;                    -- async transmit fifo empty
signal rxa_full        : std_logic;                    -- async recieve fifo full
signal rxa_empty       : std_logic;                    -- async receive fifo empty
signal xmit_enb        : std_logic := '0';             -- transmit enable
signal rcv             : std_logic;                    -- receive bit
signal xmit            : std_logic := '0';             -- transmit bit
signal rx_rd           : std_logic;                    -- receive FIFO read enable
signal rx_dout : std_logic_vector(7 downto 0);	
signal RHR_ready       : std_logic := '0';                    -- receive hold register ready
signal txa_empty_sig   : std_logic := '0';                    -- transmit FIFO empty
signal r_we            : std_logic;
signal r_read_rhr, read_rhr : std_logic;
signal rcv_cts         : std_logic := '0';
signal version_reg     : std_logic_vector(15 downto 0);
signal ver_rd          : std_logic;
signal MSR_read        : std_logic := '0';  -- flag MSR as read to clear "changed" bits
signal r_MSR_read      : std_logic := '0';

signal rd_int_en       : std_logic := '0';
signal timer_fired     : std_logic := '0';
signal timer_clk_en    : std_logic := '0';
constant DIVISOR_SETTINGS : bus4_vector(3 downto 0) := (x"E",x"9",x"9",x"9"); -- divide by 15000 (15x10x10x10)

signal xmit_cts_m1 : std_logic; -- added for metastability
signal xmit_cts    : std_logic;
signal dsr_meta, dsr_in : std_logic := '0';
signal ri_meta, ri_in : std_logic := '0';
signal dcd_meta, dcd_in : std_logic := '0';

signal paritybitstart : std_logic_vector(3 downto 0) := x"9";
signal stopbitstart   : std_logic_vector(3 downto 0) := x"A";
signal stopbitend     : std_logic_vector(3 downto 0) := x"B";
signal rcv_out        : std_logic_vector(7 downto 0) := x"00";
signal par_1, parity  : std_logic := '1';
begin

version : core_version
   port map(
      clk           => clk,                  -- system clock
      rd            => ver_rd,               -- read enable
      ID            => CORE_ID,              -- assigned ID number, 0xFF if unassigned
      version_major => CORE_VERSION_MAJOR,   -- major version number 1-15
      version_minor => CORE_VERSION_MINOR,   -- minor version number 0-15
      year          => CORE_YEAR,            -- year since 2000
      month         => CORE_MONTH,           -- month (1-12)
      day           => CORE_DAY,             -- day (1-31)
      ilevel        => i_ilevel,
      ivector       => i_ivector,
      o_data        => version_reg
      );

read_mux_regs : process(clk) is
begin
	if clk'event and clk='1' then
		ver_rd <= '0';
		read_rhr <= '0';
		MSR_read <= '0';
	
		r_rd_en <= i_rd_en;
		r_read_rhr <= read_rhr;

		LSR(6 downto 5) <= txa_empty & txa_empty;
		LSR(1 downto 0) <= txa_full & RHR_ready;
		
		if rxa_empty = '0' and RHR_ready = '0' then
			rx_rd <= '1';
			RHR_ready <= '1';
			RHR <= rx_dout;
		elsif r_read_rhr = '1' and read_rhr = '0' then
			rx_rd <= not rxa_empty;
			RHR_ready <= not rxa_empty;
			if (rxa_empty = '0') then
				RHR <= rx_dout;
			end if;
		else
			rx_rd <= '0';
		end if;
			
		if i_cs = '0' then
			o_DBus <= (others=>'0');
		else
			case i_ABus is
				when "000000" =>   
					o_DBus <=  version_reg;
					ver_rd <= i_rd_en;
				when "000001" =>	
					o_DBus <= x"00" & RHR;
					read_rhr <= i_rd_en and i_be(0);
				when "000010" =>	
					o_DBus <= LCR & MSR;
					MSR_read <= i_rd_en and i_be(0);
				when "000100" =>	
					o_DBus <= DLH & DLL;
				when "000101" =>	
					o_DBus <= IER & EFR;
				when "000110" =>	
					o_DBus <= LSR & MCR;
				when others =>	
					o_DBus <= (others=>'1');
			end case;
		end if;
	end if;
end process read_mux_regs;

--* Decode register write requests.
--/
wr_ctl_reg : process(clk)
begin
	if clk='1' and clk'event then
		r_txa <= txa;
		r_we  <= i_wr_en;

		if txa = '1' and r_txa = '0' then
			txa_we <= '1';
		else
			txa_we <= '0';
		end if;

		txa <= '0';
		FCR(2) <= '0';
		FCR(1) <= '0';
		if i_cs = '1' and i_wr_en = '1' then
			case i_ABus is
				when "000001" => 
					THR <= i_DBus(7 downto 0);
					txa <= i_be(0);
				when "000010" =>
					if i_be(0) = '1' then
						FCR(2 downto 1) <= i_DBus(2 downto 1);
					end if;
					if i_be(1) = '1' then
						LCR(6 downto 0) <= i_DBus(14 downto 8);
					end if;   
				when "000100" =>
					if i_be(1)='1' then
						DLH <= i_DBus(15 downto 8);
					end if;
					if i_be(0)='1' then
						DLL <= i_DBus(7 downto 0);
					end if;	
				when "000101" =>
					if i_be(0)='1' then
						EFR(7 downto 6) <= i_DBus(7 downto 6);
					end if;
					if i_be(1) = '1' then
						IER(0) <= i_DBus(8);
						IER(1) <= i_DBus(9);
						IER(2) <= i_DBus(10);
					end if;
				when "000110" =>	
					if i_be(0) = '1' then
						MCR(0) <= i_DBus(0);
						MCR(1) <= i_DBus(1);
						MCR(4) <= i_DBus(4);
					end if;
				when others => 
					NULL;
			end case;
		end if;
	end if;
end process wr_ctl_reg;

data_bits : process(clk)
begin
    if rising_edge(clk) then

        case LCR(1 downto 0) is
            when "00" => -- 5 data bits
                paritybitstart <= x"6";
            when "01" => -- 6 data bits
                paritybitstart <= x"7";
            when "10" => -- 7 data bits
                paritybitstart <= x"8";
            when "11" => -- 8 data bits
                paritybitstart <= x"9";
            when others => NULL;
        end case;
        
        if LCR(3) = '1' then
            stopbitstart <= paritybitstart+"0001";
        else
            stopbitstart <= paritybitstart;
        end if;
        
        if LCR(2) = '1' then
            stopbitend <= stopbitstart+"0001";
        else
            stopbitend <= stopbitstart;
        end if;
    end if;
end process data_bits;

--------------------------------------------------------------------------------

o_irq <= (IER(0) and rd_int_en) or 
         (IER(1) and txa_empty_sig) or
         (IER(2) and (MSR(1) or MSR(2) or MSR(3)));

divide_15000: srl_counter
   generic map (
      NUM_STAGES => 4
   )
   port map (
      clk        => clk,
      clk_en     => timer_clk_en,
      DIVISIONS  => DIVISOR_SETTINGS
   );

rd_int_en_proc : process(clk)
begin
   if clk'event and clk='1' then
      -- if the receive FIFO has been cleared, clear the interrupt
      if RHR_ready='0' then
         rd_int_en   <= '0';
         timer_fired <= '0';
      -- else if the receive FIFO is 1/2 full, set the interrupt (no matter what!)
      elsif rx_fifo_count(6)='1' then 
         rd_int_en   <= '1';
      -- else if there is a byte waiting to be read, look for the first clock
      elsif timer_clk_en='1' and timer_fired='0' then
         timer_fired <='1';
      elsif timer_clk_en='1' and timer_fired='1' then
         rd_int_en   <= '1';
         timer_fired <= '0';
      end if;
   end if;
end process rd_int_en_proc;

reg_txa_empty_proc : process(clk)
begin
	if (clk'event and clk = '1') then
		txa_empty_sig <= txa_empty;
	end if;
end process reg_txa_empty_proc;	

full_baud <= DLH & DLL;

reg_rcv_proc : process(clk)
begin
	if (clk'event and clk = '1') then
		if (MCR(4) = '0') then
			rcv <= i_rcv;
		else
			rcv <= xmit;
		end if;
	end if;
end process reg_rcv_proc;
--rcv <= i_rcv when MCR(4) = '0' else xmit;
o_xmit <= xmit;

o_dtr <= not MCR(0);

tx_baud_counter_proc: process(clk)
begin
	if (clk'event and clk  = '1') then
		if  xmit_active = '0'  then
			tbaud_counter <= (others => '0');
			tbaud_tc <= '0';
		elsif (tbaud_counter = full_baud) then -- 0x27f= 640.
			tbaud_counter <= (others => '0');
			tbaud_tc <= '1';
		else
			tbaud_counter <= tbaud_counter + "0000000000000001";
			tbaud_tc <= '0';
		end if;
	end if;
end process tx_baud_counter_proc;

rx_baud_counter_proc: process(clk)
begin
   if (clk'event and clk  = '1') then
		if rcv_det = '1' then
			rbaud_tc <= '0';
			rbaud_counter <= (others => '0');
			rbaud_hc <= '0';
		elsif (rbaud_counter = full_baud) then
			rbaud_counter <= (others => '0');
			rbaud_tc <= '1';
			rbaud_hc <= '0';
		elsif (rbaud_counter = full_baud(14 downto 1)) then --half_baud then -- TODO: not sure if this comparison is right...
			rbaud_counter <= rbaud_counter + "0000000000000001";
			rbaud_tc <= '0';
			rbaud_hc <= '1';
		else
			rbaud_counter <= rbaud_counter + "0000000000000001";
			rbaud_tc <= '0';
			rbaud_hc <= '0';
		end if;
	end if;
end process rx_baud_counter_proc;

receive_detector : process(clk)
begin
   if (clk'event and clk = '1') then
      rcv_meta <= rcv;

      if rcv_meta = '1' then
         rcv_det1 <= '0';               --reset to '0' for smaller logic chain
         rcv_det2 <= '0';
      else
         rcv_det1 <= '1';
         rcv_det2 <= rcv_det1;
      end if;
      
      -- start bit detected
      if rcv_det1 = '1' and rcv_det2 = '0' and  rcv_active = '0' then
         rcv_det <= '1';                -- one clock pulse @ begin of receive
      else
         rcv_det <= '0';
      end if;
   end if;
end process receive_detector;

tx_bit_count_proc: process(clk)
begin
   if (clk'event and clk  = '1') then
      xmit_cts_m1 <= i_xmit_cts; -- add FFs for metastability
      xmit_cts <= xmit_cts_m1;
      -- don't start transmit process until cts is asserted (active low)
      if (txa_empty = '0' and xmit_active = '0' and (EFR(7) = '0' or xmit_cts = '0') ) then
         xmit_active <= '1';
      elsif tbitcnt = stopbitend and tbaud_tc = '1' then
         xmit_active <= '0';
      end if;
      if tbaud_tc = '1' then
         tbitcnt <= tbitcnt + "0001";
      elsif xmit_active = '0' then
         tbitcnt <= (others => '0');
      else
         tbitcnt <= tbitcnt;
      end if;
      -- don't start transmit process until cts is asserted (active low)
      if ( txa_empty = '0' and xmit_active = '0' and (EFR(7) = '0' or xmit_cts = '0') ) then
         xmit_rd_en <= '1';
	 xmit_byte <= s_xmit_dout;
      else
         xmit_rd_en <= '0';
      end if;
   end if;
end process tx_bit_count_proc;

rx_bit_count_proc: process(clk)
begin
	if (clk'event and clk  = '1') then
		if rcv_det = '1' then
			rcv_active <= '1';
		elsif rbitcnt = "0000" and rcv_det2 = '0' and 
			(rbaud_counter = full_baud(15 downto 2)) then -- TODO: not sure if this comparison is right...
			rcv_active <= '0';
		elsif rbitcnt = stopbitstart and rbaud_hc = '1' then
			rcv_active <= '0';
		end if;
		if  rcv_active = '0' then
			rbitcnt <= (others => '0');
		elsif rbaud_hc = '1' then
			rbitcnt <= rbitcnt + "0001";
		else
			rbitcnt <= rbitcnt;
		end if;
	end if;
end process rx_bit_count_proc;

out_parity : process(clk)
begin
   if rising_edge(clk) then
       if (tbaud_tc = '1' and tbitcnt = "0000" and xmit_active = '1')  then
           par_1 <= (xmit_byte(0) xor
                      xmit_byte(1) xor
                      xmit_byte(2) xor
                      xmit_byte(3) xor
                      xmit_byte(4) xor
                      xmit_byte(5) xor
                      xmit_byte(6) xor
                      xmit_byte(7));
       end if;
       case LCR(5 downto 4) is -- Stick Parity + Even Parity Select
           when "00" =>
               parity <= not par_1;
           when "01" =>
               parity <= par_1;
           when "10" =>
               parity <= '1';
           when "11" =>
               parity <= '0';
           when others => NULL;
       end case;
   end if;
end process out_parity;

tx_shift_reg_proc:process  (clk)
begin
	if (clk'event and clk  = '1') then
		if (tbaud_tc = '1' and tbitcnt = "0000" and xmit_active = '1')  then
			tshift_reg <= xmit_byte;
		elsif tbaud_tc = '1' then
			tshift_reg <= '0' & tshift_reg(7 downto 1);
		end if;
		if xmit_active = '1' then
			xmit_enb <= '1';
			if tbitcnt=x"0" then
			    xmit <= '0';
			elsif LCR(3) = '1' and tbitcnt = paritybitstart then
			    xmit <= parity;
			elsif tbitcnt >= stopbitstart then
			    xmit <= '1';
			else 
			    xmit <= tshift_reg(0);
			end if;
		else
			xmit_enb <= '0';
			xmit <= '1'; -- stop bit
		end if;
	end if;
end process tx_shift_reg_proc;

o_xmit_enb <= xmit_enb;

rx_shift_reg_proc:process  (clk)
begin
	if (clk'event and clk  = '1') then
		if rbaud_hc = '1' then
			rshift_reg <= rcv & rshift_reg(7 downto 1);
		end if;
		
        case LCR(1 downto 0) is
            when "00" => -- 5 data bits
                rcv_out <= "000" & rshift_reg(7 downto 3);
            when "01" => -- 6 data bits
                rcv_out <= "00" & rshift_reg(7 downto 2);
            when "10" => -- 7 data bits
                rcv_out <= "0" & rshift_reg(7 downto 1);
            when "11" => -- 8 data bits
                rcv_out <= rshift_reg;
            when others => NULL;
        end case;
		
	end if;
end process rx_shift_reg_proc;

rx_flow_ctrl: process (clk)
begin
   if clk'event and clk='1' then   
      if EFR(6)='0' then
         rcv_cts <= not MCR(1);
      -- high water mark is 60 bytes
      elsif rx_fifo_count = "0111100" then
         rcv_cts <= '1';
      -- low water mark is 32 bytes
      elsif rx_fifo_count(6 downto 5) = "00" then
         rcv_cts <= '0';
      end if;
   end if;
end process rx_flow_ctrl;

rcv_wr_en <= '1' when (rbitcnt = paritybitstart and rbaud_tc = '1' and rcv_active = '1')
		else '0';

o_rcv_cts <= rcv_cts;

modem_status: process(clk)
begin
   if clk'event and clk='1' then
      r_MSR_read <= MSR_read;
      dcd_meta <= i_dcd;
      dcd_in <= dcd_meta;
      ri_meta <= i_ri;
      ri_in <= ri_meta;
      dsr_meta <= i_dsr;
      dsr_in <= dsr_meta;
      
      MSR(7) <= not dcd_in;
      MSR(6) <= not ri_in;
      MSR(5) <= not dsr_in;
      MSR(4) <= not xmit_cts;
      if MSR_read = '0' and r_MSR_read = '1' then	 -- don't clear until end of cycle
         MSR(3 downto 0) <= "0000";
      else
          if MSR(7) = dcd_in then
             MSR(3) <= '1';
          end if;
          if MSR(6) = ri_in then
             MSR(2) <= '1';
          end if;
          if MSR(5) = dsr_in then 
             MSR(1) <= '1';
          end if;
          if MSR(4) = xmit_cts then
             MSR(0) <= '1';
          end if;
      end if;
   end if;
end process modem_status;

--parity_in <= 	vsc_fifo_in(7) xor vsc_fifo_in(6) xor
--					vsc_fifo_in(5) xor vsc_fifo_in(4) xor
--					vsc_fifo_in(3) xor vsc_fifo_in(2) xor
--					vsc_fifo_in(1) xor vsc_fifo_in(0) ;

fifo_from_fpga2driver : fifo_dram_sync
		Generic map (
			WIDTH => 8,
			DEPTH_BITS => 6,
			RAM_STYLE_HINT => "distributed"
		)
		port map (
			clk        => clk,
			i_reset    => FCR(2),
			i_din      => THR,
			i_wr_en    => txa_we,
			i_rd_en    => xmit_rd_en,
			o_dout     => s_xmit_dout, --xmit_byte,
			o_full     => txa_full,
			o_empty    => txa_empty,
			o_almost_full => open,
			o_almost_empty => open,
			o_count    => tx_fifo_count);

fifo_from_recvr2fpga : fifo_dram_sync
		Generic map (
			WIDTH => 8,
			DEPTH_BITS => 6,
			RAM_STYLE_HINT => "distributed"
		)
		port map (
			clk        => clk,
			i_reset    => FCR(1),
			i_din      => rcv_out, 
			i_wr_en    => rcv_wr_en,
			i_rd_en    => rx_rd,
			o_dout     => rx_dout, --RHR,
			o_full     => rxa_full,
			o_almost_full => open,
			o_almost_empty => open,
			o_empty    => rxa_empty,
			o_count    => rx_fifo_count);

end rtl;
