--- Title: ADS8329_ext.vhd
--- Description: Extensible Core for ADS8329 ADC.
---
---     o  0
---     | /       Copyright (c) 2007-2013
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 3/12/2007
--- Version: 1.04
--- Revisions:  
--- 1.00 Baseline
--- 1.01 Map FIFO also to 0x40 through 0x7F to support MityDSP PRO
--- 1.02 Fix capture length bug
--- 1.03 Fix First word error bug
--- 1.04 Add capability to soft reset device
--- 1.05 Migrate to OMAP-L138 (~100 MHZ EMIFA)

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
library UNISIM;
use UNISIM.VCOMPONENTS.ALL;
library WORK;
use WORK.MityDSP_L138_pkg.ALL;

entity ADS8329_ext is
   generic
   (
      FIFO_DEPTH_TWO_TO_N : integer range 9 to 15 := 10 -- 10 is 1K words
   );
   port 
   ( 
      clk             : in  std_logic;
      i_ABus          : in  std_logic_vector(5 downto 0);
      i_DBus          : in  std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in  std_logic;
      i_rd_en         : in  std_logic;
      i_cs            : in  std_logic;	
      o_irq           : out std_logic;
      i_ilevel        : in std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      i_ivector       : in std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)

      -- External 32-bit First-Word-Fall-Through FIFO Write Ports
      o_fifo_din      : out std_logic_vector(31 downto 0);                    -- fifo's data in
      o_fifo_wr_stb   : out std_logic;                                        -- fifo's write strobe (1 clk wide)
      o_fifo_wr_clk   : out std_logic;                                        -- fifo's write clock
      -- External 32-bit First-Word-Fall-Through FIFO Read Ports
      o_fifo_rd_stb   : out std_logic;                                        -- fifo's read strobe (1 clk wide)
      o_fifo_rd_clk   : out std_logic;                                        -- fifo's read clock
      i_fifo_dout     : in  std_logic_vector(31 downto 0);                    -- fifo's data out
      i_fifo_full     : in  std_logic;                                        -- fifo's full flag
      i_fifo_empty    : in  std_logic;                                        -- fifo's empty
      i_fifo_rd_count : in  std_logic_vector(FIFO_DEPTH_TWO_TO_N-1 downto 0); -- fifo's read count

      -- ADC control / data ports
      i_adc_clk      : in  std_logic;
      i_udata        : in  std_logic_vector(4 downto 0);   -- User Data lines
      i_trigger      : in  std_logic;                      -- external trigger
      o_adc_convst_n : out std_logic;                      -- adc convert start strobe      
      i_adc_eoc      : in  std_logic;                      -- end of convert signal from ADC
      o_adc_sclk     : out std_logic;                      -- ADC data shift clock (1/4 EMIF clock)
      o_adc_sdi      : out std_logic;                      -- ADC SPI input line
      i_adc_sdo      : in  std_logic;                      -- ADC SPI output line
      o_adc_cs_n     : out std_logic;                      -- ADC frame sync line
      i_fifo_re      : in  std_logic;                      -- optional FIFO read enable (for CE 2 access)
      o_fifo_data    : out std_logic_vector(31 downto 0)   -- optional FIFO output enable (for CE 2 access)
    );
end ADS8329_ext;

--*
--* @short Register Transfer Logic Implementation of ADS8329_ext entity.
--* 
--/
architecture rtl of ADS8329_ext is

-- All Used components should be declared first.
--

signal version_reg       : std_logic_vector(15 downto 0) := (others=>'0');  -- version register
signal ver_rd            : std_logic := '0';

-- capture and control status register fields
signal ccsr_en           : std_logic := '0';   -- enable 
signal ccsr_md           : std_logic := '0';   -- mode
signal ccsr_ext          : std_logic := '0';   -- external trigger enable
signal ccsr_bc           : std_logic := '0';   -- burst complete
signal ccsr_pack         : std_logic := '0';   -- pack data samples
signal ccsr_fb           : std_logic := '0';   -- bit flip control
signal bc_clear          : std_logic := '0';   -- used to clear burst complete bit
signal bc_clear_r1       : std_logic := '0';

-- burst capture delay register fields
signal bdr_bcd           : std_logic_vector(15 downto 0);  -- burst capture delay register

-- burst capture quantity register fields
signal bqr_bcq            : std_logic_vector(15 downto 0);  -- burst capture quantity register

-- FIFO status register fields
signal fsr_lvl           : std_logic_vector(FIFO_DEPTH_TWO_TO_N-1 downto 0);
signal fsr_f             : std_logic := '0';                     -- fifo full
signal fsr_tqf           : std_logic := '0';                     -- fifo three quarters full
signal fsr_hf            : std_logic := '0';                     -- fifo half full
signal fsr_qf            : std_logic := '0';                     -- fifo quarter full
signal fsr_e             : std_logic := '0';                     -- fifo empty

-- interrupt enable register fields
signal ier_bcie          : std_logic := '0';  -- burst complete ie
signal ier_fie           : std_logic := '0';  -- fifo full ie
signal ier_tqfie         : std_logic := '0';  -- three quarters ie
signal ier_hfie          : std_logic := '0';  -- half full ie
signal ier_qfie          : std_logic := '0';  -- quarter full ie

-- FIFO control and data lines
signal fifo_write        : std_logic := '0';
signal fifo_read         : std_logic := '0';
signal fifo_in           : std_logic_vector(31 downto 0) := (others=>'0');  -- fifo data input
signal fifo_out          : std_logic_vector(31 downto 0) := (others=>'0');  -- fifo data output

-- burst capture logic 
signal counter           : std_logic_vector(15 downto 0) := (others=>'0');

signal temp_count          : std_logic_vector(4 downto 0)  := (others=>'0');
constant CONVST_CNT_CLOCKS : std_logic_vector(4 downto 0)  := "01110"; -- Assumes ~100 MHz EMIFA clock
constant DATA_CLOCKS       : std_logic_vector(4 downto 0)  := "01111";
constant ADC_READ_WORD     : std_logic_vector(15 downto 0) := "1101111111111111"; 
constant ADC_CONFIG_WORD   : std_logic_vector(15 downto 0) := "1110111111111111";
signal   adcdata           : std_logic_vector(15 downto 0) := (others=>'0');
signal   adcoutdata        : std_logic_vector(15 downto 0) := (others=>'0');

signal cfg_wd : std_logic_vector(15 downto 0) := ADC_CONFIG_WORD;
signal cfg_wd_written : std_logic := '0';
signal cfg_wd_toggle : std_logic := '0';
signal cfg_wd_toggle_r : std_logic := '0';

-- miscellaneous process signals
signal word_toggle        : std_logic := '0';  -- used in packed mode
signal reset              : std_logic := '0';
signal read_en            : std_logic := '0';
signal first_skipped      : std_logic := '0';

signal i_trigger_r1       : std_logic := '0';
signal i_trigger_r2       : std_logic := '0';
signal i_trigger_r3       : std_logic := '0';

signal i_adc_clk_r1  : std_logic := '0';
signal i_adc_clk_r2  : std_logic := '0';
signal i_adc_clk_r3  : std_logic := '0';

signal adc_eoc : std_logic := '0';

type CAPTURE_STATES is (
      IDLE,            -- ADC is idle
      DELAY,
      ASSERT_CONVST,
      LOAD_DATA,
      STORE_DATA,
      WAIT_ON_CLK);
signal sys_mode : CAPTURE_STATES := IDLE;

signal configured         : std_logic := '0';
signal out_clk            : std_logic_vector(1 downto 0) := "00";

constant ID            : std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR( 24, 8); -- assigned ID number, 0xFF if unassigned
constant version_major : std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR(  1, 4); -- major version number 1-15
constant version_minor : std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR(  5, 4); -- minor version number 0-15
constant year          : std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 13, 5); -- year since 2000
constant month         : std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 05, 4); -- month (1-12)
constant day           : std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 01, 5); -- day (1-31)


begin -- architecture: rtl of ADS8329_ext

reset_on_cfg: roc port map (O => reset);

----------------------------------------
-- Version History
-- 1.0 - Baseline
version : core_version
   port map(
      clk           => clk,            -- system clock
      rd            => ver_rd,         -- read enable
      ID            => ID,             -- assigned ID number, 0xFF if unassigned
      version_major => version_major,  -- major version number 1-15
      version_minor => version_minor,  -- minor version number 0-15
      year          => year,           -- year since 2000
      month         => month,          -- month (1-12)
      day           => day,            -- day (1-31)
      ilevel        => i_ilevel,
      ivector       => i_ivector,
      o_data        => version_reg
      );

o_fifo_din    <= fifo_in;
o_fifo_wr_stb <= fifo_write;
o_fifo_wr_clk <= clk;
o_fifo_rd_stb <= fifo_read or i_fifo_re;
o_fifo_rd_clk <= clk;
fifo_out      <= i_fifo_dout;
fsr_f         <= i_fifo_full;
fsr_e         <= i_fifo_empty;
fsr_lvl       <= i_fifo_rd_count;
fsr_tqf       <= '1' when fsr_lvl(FIFO_DEPTH_TWO_TO_N-1 downto FIFO_DEPTH_TWO_TO_N-2)="11" else '0';
fsr_hf        <= fsr_lvl(FIFO_DEPTH_TWO_TO_N-1);
fsr_qf        <= fsr_lvl(FIFO_DEPTH_TWO_TO_N-1) or fsr_lvl(FIFO_DEPTH_TWO_TO_N-2);

o_irq <= (ier_bcie  and ccsr_bc)   or -- busrt complete
         (ier_fie   and fsr_f)     or -- fifo full
         (ier_tqfie and fsr_tqf)   or -- three quarters full
         (ier_hfie  and fsr_hf)    or -- half full
         (ier_qfie  and fsr_qf);      -- quarter full

o_fifo_data <= i_fifo_dout;
o_adc_sclk  <= out_clk(1);
o_adc_sdi   <= adcoutdata(15);

--* Handle read requests from the processor
--/
read_mux_regs : process (clk) is
begin
	if clk'event and clk='1' then
		if (i_ABus="001101" or (i_ABus(5)='1' and i_ABus(0)='1')) and i_cs='1' and i_rd_en='1' then
			fifo_read <= '1';
		else
			fifo_read <= '0';
		end if;
	  
		-- toggle version register 
		ver_rd <= '0';
		
		adc_eoc <= i_adc_eoc;
	    
		-- address decoding
		if i_cs = '0' then
			o_DBus <= (others=>'0');
		else
			case i_ABus is
				when "000000" =>   
					o_DBus <=  version_reg;
					ver_rd <= i_rd_en;

				when "000010" =>   
					o_DBus <= adc_eoc & x"00" & ccsr_fb & ccsr_pack & '0' & ccsr_bc & ccsr_ext & ccsr_md & ccsr_en;

				when "000100" =>	
					o_DBus <= bdr_bcd;

				when "000110" =>	
					o_DBus <= bqr_bcq;

				when "001000" => 
					o_DBus <= x"00" & "000" & fsr_f & fsr_tqf & fsr_hf & fsr_qf & fsr_e; 
				when "001001" => 
					o_DBus <= CONV_STD_LOGIC_VECTOR(0, 16-FIFO_DEPTH_TWO_TO_N) & fsr_lvl;

				when "001010" =>	
					o_DBus <= x"00" & "000" & ier_bcie & ier_fie & ier_tqfie & ier_hfie & ier_qfie;

				when "001100" =>	
					o_DBus <= fifo_out(15 downto 0);
				when "001101" =>	
					o_DBus <= fifo_out(31 downto 16);

				when "001110" =>
					o_DBus <= CONV_STD_LOGIC_VECTOR(2**FIFO_DEPTH_TWO_TO_N,16);  -- FIFO DEPTH
				when "001111" =>
					o_DBus <= CONV_STD_LOGIC_VECTOR((2**FIFO_DEPTH_TWO_TO_N)/(2**16),16);  -- FIFO DEPTH

				when "010000" =>
					o_DBus <= cfg_wd;
				when "010001" =>
					o_DBus <= x"000" &"000" & configured;

				when "1----0" =>
					o_DBus <= fifo_out(15 downto 0);
				when "1----1" =>
					o_DBus <= fifo_out(31 downto 16);

				when others =>	
					o_DBus <= (others=>'X');
			end case;
		end if;
	end if;
end process read_mux_regs;

--* Decode register write requests.
--/
wr_ctl_reg : process(clk)
begin
	if clk='1' and clk'event then
      
		if i_ABus="000010" and i_cs='1' and i_wr_en='1' and i_DBus(3)='1' then
			bc_clear <= not bc_clear;  -- toggle
		end if;
      
		if i_cs = '1' and i_wr_en = '1' then
			case i_ABus is
				when "000010" =>  -- ccsr access
					ccsr_fb    <= i_DBus(6);
					ccsr_pack  <= i_DBus(5);
					ccsr_ext   <= i_DBus(2);
					ccsr_md    <= i_DBus(1);
					ccsr_en    <= i_DBus(0);

				when "000100" =>	
					bdr_bcd <= i_DBus(15 downto 0);

				when "000110" =>
					bqr_bcq <= i_DBus(15 downto 0);

				when "001010" =>	
					ier_bcie  <= i_DBus(4);
					ier_fie   <= i_DBus(3);
					ier_tqfie <= i_DBus(2);
					ier_hfie  <= i_DBus(1);
					ier_qfie  <= i_DBus(0);

				when "010000" =>
					cfg_wd <= i_DBus(15 downto 0);
					cfg_wd_toggle <= not cfg_wd_toggle;

				when others => 
					NULL;
			end case;
		end if;
	end if;
end process wr_ctl_reg;

--* main shift / sampling logic
--  The timing design for this clock is from the Read While Converting 
--  mode.  (figure 3 of the specification).  Because of this, the 
--  sample clocked out is actually N-1 for the Nth convert start
--  strobe.
--/
state_proc : process(clk, reset)
begin
   if reset='1' then
      sys_mode   <= IDLE;
      o_adc_cs_n <= '1';
      out_clk    <= (others=>'0');
      configured <= '0';
      first_skipped <= '0';
      word_toggle <= '0';
   elsif clk'event and clk='1' then
      i_trigger_r1 <= i_trigger;
      i_trigger_r2 <= i_trigger_r1;
      i_trigger_r3 <= i_trigger_r2;
      i_adc_clk_r1 <= i_adc_clk;
      i_adc_clk_r2 <= i_adc_clk_r1;
      i_adc_clk_r3 <= i_adc_clk_r2;
      bc_clear_r1 <= bc_clear;

      fifo_write <= '0';
      read_en <= '0';
				
		if cfg_wd_toggle_r /= cfg_wd_toggle then
			configured <= '0';
		end if;
      
		case sys_mode is
         when IDLE =>
            -- transitions must occur on rising edge of convert start strobe
            counter     <= (others => '0');
            temp_count  <= (others => '0');
            word_toggle <= '0';
            first_skipped <= '0';
            if i_adc_clk_r3='0' and i_adc_clk_r2='1' then
					if cfg_wd_toggle_r /= cfg_wd_toggle then -- if need to reconfig, just go...
						sys_mode <= ASSERT_CONVST;
               elsif ccsr_en='1' then    -- enabled
                  if ccsr_md='1' then -- continuous wave (no delay)
                     if ccsr_ext='0' then -- no external trigger, go
                        sys_mode <= ASSERT_CONVST;
                     elsif i_trigger_r2='1' then
                        sys_mode <= ASSERT_CONVST; 
                     end if;
                  else                -- burst capture (delay needed)
                     -- for internal trigger, require bc to be cleared first
                     -- or external trigger trigger seen
                     if (ccsr_ext='0' and ccsr_bc='0') or
                        (ccsr_ext='1' and i_trigger_r2='1')
                     then 
                        if bdr_bcd /= x"0000" then
                            counter <= counter+'1';
                            sys_mode <= DELAY;
                        else
                            sys_mode <= ASSERT_CONVST;
                        end if;
                     end if;
                  end if;
               end if;
            end if;
            
         when DELAY =>
            -- wait n clocks (busy transitions) then go to converting
            if i_adc_clk_r3='0' and i_adc_clk_r2='1' then
               if counter = bdr_bcd then
                  counter <= (others => '0');
                  sys_mode <= ASSERT_CONVST;
               else
                  counter <= counter+'1';
               end if;
            end if;
            
         when ASSERT_CONVST =>
			   -- if ADC is configured, then go ahead and do a read cycle
            if configured='1' then
               adcoutdata <= ADC_READ_WORD;
            else
               adcoutdata <= cfg_wd;
					cfg_wd_toggle_r <= cfg_wd_toggle;
            end if;
            -- wait for convert start hold time requirements to be met
            if temp_count=CONVST_CNT_CLOCKS then
               o_adc_convst_n <= '1';
               o_adc_cs_n <= '0';
               temp_count <= (others=>'0');
               sys_mode <= LOAD_DATA;
            else
               o_adc_convst_n <= '0';
               temp_count <= temp_count+'1';
            end if;
            
         when LOAD_DATA =>
            out_clk <= out_clk+'1';
            if out_clk="01" then
               -- don't shift output word on first clock
               if temp_count /= x"0" then
                  adcoutdata <= adcoutdata(14 downto 0) & '0';
               end if;
               -- shift input word
               adcdata <= adcdata(14 downto 0) & i_adc_sdo;
            elsif out_clk="11" then
               -- if we aren't done
               if temp_count /= DATA_CLOCKS then
                  temp_count <= temp_count+'1';
               else
                  temp_count <= (others=>'0');
                  sys_mode <= STORE_DATA;
               end if;
            end if;
            
         when STORE_DATA =>
            -- if we are packing and we've skipped our first sample
            if ccsr_pack = '1' and first_skipped = '1' then -- pack the data
               word_toggle <= not word_toggle;
               if word_toggle='0' then
                  fifo_in(15)            <= ccsr_fb xor adcdata(15);
                  fifo_in(14 downto 0)   <= adcdata(14 downto 0);
               else
                  fifo_in(31) <= ccsr_fb xor adcdata(15);
                  fifo_in(30 downto 16)  <= adcdata(14 downto 0);
                  fifo_write             <= '1'; -- '1' when we should clock out data
               end if;
            else  -- single sample 
               fifo_in(31 downto 22)     <= (others=>'0');
               fifo_in(21 downto 17)     <= i_udata;
               fifo_in(16)               <= '0';
               fifo_in(15)               <= ccsr_fb xor adcdata(15);
               fifo_in(14 downto 0)      <= adcdata(14 downto 0);
               fifo_write                <= first_skipped; -- '1' when we should clock out data
            end if;
            sys_mode <= WAIT_ON_CLK;
            -- ensure we flag that we've skipped at least one sample after we've been configured
            first_skipped <= configured;             
            
         when WAIT_ON_CLK =>
            o_adc_cs_n <= '1';
            -- at this point the ADC should be configured if not held in reset
            configured <= '1';
            -- if we wrote to the FIFO, bump up the FIFO words written count
            if fifo_write='1' then
               counter <= counter+'1';
            end if;
            -- if we've been disabled, or we've written out the requested capture quantity
				-- or we've requested an ADC Reset
            if ccsr_en='0' or (ccsr_md='0' and fifo_write='1' and counter=bqr_bcq) then
               sys_mode <= IDLE;
            -- else our input clock reference has fired, start next aquisition
            elsif  i_adc_clk_r3='0' and i_adc_clk_r2='1' then
               sys_mode <= ASSERT_CONVST;
            end if;
            
         when others => NULL;
      end case;
      
      -- would this be better in the state machine? 
      if bc_clear_r1 /= bc_clear then -- if toggled, clear
         ccsr_bc <= '0';
      -- else if WAITING and a FIFO was written to meet quantity, mark complete
      elsif sys_mode=WAIT_ON_CLK and ccsr_md='0' and fifo_write='1' and counter=bqr_bcq then
         ccsr_bc <= '1';
      end if;
   end if;
end process state_proc;

end rtl;
