/**
 * \file    demoMain.c
 *
 * \brief   This file contains the main() and other functions.
 */

/*
* Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com/ 
*
*  Redistribution and use in source and binary forms, with or without 
*  modification, are permitted provided that the following conditions 
*  are met:
*
*    Redistributions of source code must retain the above copyright 
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the 
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "hw_types.h"
#include "interrupt.h"
#include "demoI2C.h"
#include "demoRaster.h"
#include "demoTouch.h"
#include "demoMain.h"
#include "soc_OMAPL138.h"
#include "demoToneLoop.h"
#include "demoCfg.h"
#include "demoAic31.h"
#include "demoTimer.h"
#include "demoLedIf.h"
#include "demoGpio.h"
#include "demoEnet.h"
#include "demoRtc.h"
#include "lcdkOMAPL138.h"
#include "uartStdio.h"
#include "psc.h"
#include "usblib.h"
#include "usbhid.h"
#include "usbhost.h"
#include "usbhhid.h"
#include "usbhhidmouse.h"
#include "delay.h"

#include "demoImages.h"
#include "demoSlides.h"

#include "grlib.h"
#include <string.h>


/****************************************************************************
**                   INTERNAL MACRO DEFINITIONS                                       
****************************************************************************/
#define NUM_OF_IMAGES           (11u)
 
/****************************************************************************
**                   LOCAL FUNCTION PROTOTYPES                                
****************************************************************************/
static void CoOrdAction(int x, int y);
static void PeripheralsSetUp(void);
static void ClickAction(void);
static void PrevContextClear(void);
static void ActionIdle(void);
static void ActionDefault(void);
static void ActionI2C(void);
static void ActionTimer(void);
static void ActionGpio(void);
static void ActionTimeSet(void);
static void ActionEnetInit(void);
static void ActionUSBMouse(void);

static void ActionMenu(void);
static void ActionMenuIntro(void);
static void ActionMenuWebDemo(void);
static void ActionMenuMcASP(void);
static void ActionMenuSPI(void);
static void ActionMenuUart(void);
static void ActionMenuGPIO(void);
static void ActionMenuRTC(void);
static void ActionMenuI2C(void);
static void ActionMenuUSBMouse(void);
static void ActionMenuEthernet(void);
static void ActionMenuTimer(void);

static void updateUSBMouseImage(void);

extern void USBInit(void);
extern void ActionTouchPad(void);
extern void ActionLeftClick(void);
extern void ActionMiddleClick(void);
extern void ActionRightClick(void);

extern unsigned int ipAddr;
extern volatile tBoolean bConnected;

extern tContext sContextBanner;
extern tContext sContextChoice;
extern tContext sContextEthernet;
extern tContext sContextGpio;
extern tContext sContextI2c;
extern tContext sContextIntro;
extern tContext sContextMcasp;
extern tContext sContextMenu;
extern tContext sContextRtc;
extern tContext sContextSpi;
extern tContext sContextTimer;
extern tContext sContextUart;
extern tContext sContextUsbMouse;

//*****************************************************************************
//
// The size of the USB host controller's memory pool in bytes.
//
//*****************************************************************************
#define HCD_MEMORY_SIZE         128

//*****************************************************************************
//
// The memory pool to provide to the USB Host controller driver.
//
//*****************************************************************************
unsigned char g_pHCDPool[HCD_MEMORY_SIZE];

//*****************************************************************************
//
// The size of the mouse device interface's memory pool in bytes.
//
//*****************************************************************************
#define MOUSE_MEMORY_SIZE       128

//*****************************************************************************
//
// The memory pool to provide to the mouse device.
//
//*****************************************************************************
unsigned char g_pucMouseBuffer[MOUSE_MEMORY_SIZE];

//*****************************************************************************
//
// Declare the USB Events driver interface.
//
//*****************************************************************************
DECLARE_EVENT_DRIVER(g_sUSBEventDriver, 0, 0, USBHCDEvents);

//*****************************************************************************
//
// The global that holds all of the host drivers in use in the application.
// In this case, only the Mouse class is loaded.
//
//*****************************************************************************
static tUSBHostClassDriver const * const g_ppHostClassDrivers[] =
{
    &g_USBHIDClassDriver
    ,&g_sUSBEventDriver
};

//*****************************************************************************
//
// This global holds the number of class drivers in the g_ppHostClassDrivers
// list.
//
//*****************************************************************************
static const unsigned int g_ulNumHostClassDrivers =
    sizeof(g_ppHostClassDrivers) / sizeof(tUSBHostClassDriver *);

//*****************************************************************************
//
// The global value used to store the mouse instance value.
//
//*****************************************************************************
static unsigned int g_ulMouseInstance;

//*****************************************************************************
//
// The global values used to store the mouse state.
//
//*****************************************************************************
static unsigned int g_ulButtons;
static tRectangle g_sCursor;

//*****************************************************************************
//
// This enumerated type is used to hold the states of the mouse.
//
//*****************************************************************************
enum
{
    //
    // No device is present.
    //
    STATE_NO_DEVICE,

    //
    // Mouse has been detected and needs to be initialized in the main
    // loop.
    //
    STATE_MOUSE_INIT,

    //
    // Mouse is connected and waiting for events.
    //
    STATE_MOUSE_CONNECTED,

    //
    // An unsupported device has been attached.
    //
    STATE_UNKNOWN_DEVICE,

    //
    // A power fault has occured.
    //
    STATE_POWER_FAULT
}
eUSBState;

int clicked = 0;

/*
** Registers USB0Host interrupts
*/
#ifdef _TMS320C6X
void USB0HostIntRegister(unsigned int cpuINT)
{
    /* Register the ISR in the Interrupt Vector Table.*/
    IntRegister(cpuINT, USB0HostIntHandler);
    IntEventMap(cpuINT, SYS_INT_USB0);
}
#else
void USB0HostIntRegister(unsigned int channel)
{
    /* Register the ISR in the Interrupt Vector Table.*/
    IntRegister(SYS_INT_USB0, USB0HostIntHandler);
    IntChannelSet(SYS_INT_USB0, channel);
}
#endif

//*****************************************************************************
//
// These defines are used to define the screen constraints to the application.
//
//*****************************************************************************
#define DISPLAY_BANNER_HEIGHT   0
#define DISPLAY_BANNER_BG       ClrDarkBlue
#define DISPLAY_BANNER_FG       ClrWhite
#define DISPLAY_MOUSE_FG        ClrBlack
#define DISPLAY_MOUSE_SIZE      2

/****************************************************************************
**                  GLOBAL VARIABLES DEFINITIONS                                         
****************************************************************************/
/*
** Coordinates for each icon.
*/
static int const xyDefault[4] =  {0, 0, 0, 0};

static int const xyNext[4] = {XMIN_NEXT, XMAX_NEXT, YMIN_NEXT, YMAX_NEXT};

static int const xyPrev[4] = {XMIN_PREV, XMAX_PREV, YMIN_PREV, YMAX_PREV};	
				
static int const xyHome[4] = {XMIN_HOME, XMAX_HOME, YMIN_HOME, YMAX_HOME};

static int const xyTimeSet[4] = {XMIN_RTC_STD, XMAX_RTC_STD,
                                 YMIN_RTC_STD, YMAX_RTC_STD};
								 
static int const xyIntro[4] =  { XMIN_INTRO, XMAX_INTRO, YMIN_INTRO, YMAX_INTRO};

static int const xyWebDemo[4] = { XMIN_WEB_DEMO, XMAX_WEB_DEMO,
								  YMIN_WEB_DEMO, YMAX_WEB_DEMO};
								  
static int const xyMcASP[4] = { XMIN_MCASP, XMAX_MCASP, YMIN_MCASP, YMAX_MCASP};

static int const xySPI[4] =  { XMIN_SPI, XMAX_SPI, YMIN_SPI, YMAX_SPI}; 

static int const xyUart[4] =  { XMIN_UART, XMAX_UART, YMIN_UART, YMAX_UART}; 

static int const xyGPIO[4] = { XMIN_GPIO, XMAX_GPIO, YMIN_GPIO, YMAX_GPIO}; 

static int const xyRTC[4] = { XMIN_RTC, XMAX_RTC, YMIN_RTC, YMAX_RTC}; 

static int const xyI2C[4] = { XMIN_I2C, XMAX_I2C, YMIN_I2C, YMAX_I2C}; 

static int const xyUSBMouse[4] = { XMIN_USB_MOUSE, XMAX_USB_MOUSE, 
								   YMIN_USB_MOUSE, YMAX_USB_MOUSE}; 
								   
static int const xyTimer[4] = { XMIN_TIMER, XMAX_TIMER, YMIN_TIMER, YMAX_TIMER};

static int const xyEthernet[4] = { XMIN_ETHERNET, XMAX_ETHERNET, 
										YMIN_ETHERNET, YMAX_ETHERNET }; 

static int const xyMouseLeft[4] = { XMIN_MOUSE_LEFT, XMAX_MOUSE_LEFT, 
										YMIN_MOUSE_LEFT, YMAX_MOUSE_LEFT }; 
										
static int const xyMouseMiddle[4] = { XMIN_MOUSE_MIDDLE, XMAX_MOUSE_MIDDLE, 
										YMIN_MOUSE_MIDDLE, YMAX_MOUSE_MIDDLE }; 

static int const xyMouseRight[4] = { XMIN_MOUSE_RIGHT, XMAX_MOUSE_RIGHT, 
										YMIN_MOUSE_RIGHT, YMAX_MOUSE_RIGHT }; 
										
static int const xyTouchPad[4] = { XMIN_TOUCH_PAD, XMAX_TOUCH_PAD, 
										YMIN_TOUCH_PAD, YMAX_TOUCH_PAD }; 

/*
** Specifications for each icon
*/
static TOUCHSPEC const touchSpecBanner[NUM_ICON_BANNER] =
                    {
                       {xyDefault, ActionIdle}
                    };

static TOUCHSPEC const touchSpecMenu[NUM_ICON_MENU] =
                    {
					   {xyIntro, 	ActionMenuIntro},
					   {xyWebDemo, 	ActionMenuWebDemo},
					   {xyMcASP, 	ActionMenuMcASP},
					   {xyUart, 	ActionMenuUart},
					   {xyGPIO, 	ActionMenuGPIO},
					   {xyRTC, 		ActionMenuRTC},
					   {xyUSBMouse, ActionMenuUSBMouse},
					   {xyTimer, 	ActionMenuTimer},
					   {xyEthernet, ActionMenuEthernet},								   
                    };
					
static TOUCHSPEC const touchSpecIntro[NUM_ICON_INTRO] = 
                    {
                       {xyNext, ActionEnetInit},
                       {xyPrev, ActionMenu},
                       {xyHome, ActionMenu}
                    };
static TOUCHSPEC const touchSpecI2C[NUM_ICON_I2C] =
                    {
                       {xyNext, ActionUSBMouse},
					   {xyPrev, ActionDefault},
					   {xyHome, ActionMenu}
                    };
static TOUCHSPEC const touchSpecUart[NUM_ICON_UART] = 
                    {
                       {xyNext, ActionGpio},
					   {xyPrev, ActionDefault},
					   {xyHome, ActionMenu}
                    };
static TOUCHSPEC const touchSpecRtc[NUM_ICON_RTC] = 
                    {
                       {xyNext, ActionUSBMouse},
					   {xyPrev, ActionGpio},
					   {xyHome, ActionMenu},
                       {xyTimeSet, ActionTimeSet}					   
                    };
static TOUCHSPEC const touchSpecDefault[NUM_ICON_DFLT] = 
                    {
                       {xyNext, ActionDefault},
                       {xyPrev, ActionDefault},
                       {xyHome, ActionMenu}
                    };
static TOUCHSPEC const touchSpecMcASP[NUM_ICON_MCASP] = 
                    {
                       {xyNext, ActionDefault},
                       {xyPrev, ActionEnetInit},
                       {xyHome, ActionMenu}
                    };					
static TOUCHSPEC const touchSpecTimer[NUM_ICON_MCASP] = 
                    {
                       {xyNext, ActionDefault},
                       {xyPrev, ActionUSBMouse},
                       {xyHome, ActionMenu}			
					};
static TOUCHSPEC const touchSpecEthernet[NUM_ICON_ENET] =
                    {
                       {xyNext, ActionMenu},
                       {xyPrev, ActionTimer},
                       {xyHome, ActionMenu}
					};				

static TOUCHSPEC const touchSpecUSBMouse[NUM_ICON_USB_MOUSE] =
                    {
                       {xyNext, ActionTimer},
                       {xyPrev, ActionDefault},
                       {xyHome, ActionMenu},
					   {xyMouseLeft, ActionLeftClick},
					   {xyMouseMiddle, ActionMiddleClick},
					   {xyMouseRight, ActionRightClick},
					   {xyTouchPad, ActionTouchPad}					   
					};				
					
/*
** Context information.
** Image, number of icons in the image, specification.
** The next icon will come as the last specification for each image.
*/
IMAGECONTEXT contextInfo[NUM_OF_IMAGES] =             
                    {
					   {bannerImage + HW_VAL, NUM_ICON_BANNER, touchSpecBanner},
                       {menuImage + HW_VAL, NUM_ICON_MENU, touchSpecMenu},
                       {introImage + HW_VAL, NUM_ICON_INTRO, touchSpecIntro},
                       {choiceImage + HW_VAL, NUM_ICON_CHOICE, touchSpecDefault},
                       {mcaspImage + HW_VAL, NUM_ICON_MCASP, touchSpecMcASP},
                       {uartImage + HW_VAL, NUM_ICON_UART, touchSpecUart},
                       {gpioImage + HW_VAL, NUM_ICON_GPIO, touchSpecDefault},
                       {rtcImage + HW_VAL, NUM_ICON_RTC, touchSpecRtc},
					   {usbMouseImage + HW_VAL, NUM_ICON_USB_MOUSE, touchSpecUSBMouse},
                       {timerImage + HW_VAL, NUM_ICON_TMR, touchSpecTimer},
                       {ethernetImage + HW_VAL, NUM_ICON_ENET, touchSpecEthernet},
                    };

/*
** The variable which is used to keep track of the current slide.
*/
volatile unsigned int imageCount = 0;

/*
** The variable which indicates which peripheral demo to be shown.
*/
unsigned int clickIdx = 0;

/*
** The variable which indicates whether ethernet is initialized.
*/
unsigned int enetInitFlag = FALSE;

/*
** The variable which indicates whether an action is called as a result of prev
** button click.
*/
unsigned int prevAction = 0;

/****************************************************************************
**                      FUNCTION DEFINITIONS                                         
****************************************************************************/
tContext * GetCurrentContext(unsigned int index)
{
    switch(index)
    {
        case 0:
        {
        	return(&sContextBanner);
        }
        case 1:
        {
        	return(&sContextMenu);
        }
        case 2:
        {
        	return(&sContextIntro);
        }
        case 3:
        {
        	return(&sContextChoice);
        }
        case 4:
        {
        	return(&sContextMcasp);
        }
        case 5:
        {
        	return(&sContextUart);
        }
        case 6:
        {
        	return(&sContextGpio);
        }
        case 7:
        {
        	return(&sContextRtc);
        }
        case 8:
        {
        	return(&sContextUsbMouse);
        }
        case 9:
        {
        	return(&sContextTimer);
        }
        case 10:
        {
        	return(&sContextEthernet);
        }

        default:
        {
        	return(GetCurrentContext(imageCount));
        }
    }
}

//*****************************************************************************
//
// This function updates the cursor position based on deltas received from
// the mouse device.
//
// \param iXDelta is the signed movement in the X direction.
// \param iYDelta is the signed movement in the Y direction.
//
// This function is called by the mouse handler code when it detects a change
// in the position of the mouse.  It will take the inputs and force them
// to be constrained to the display area of the screen.  If the left mouse
// button is pressed then the mouse will draw on the screen and if it is not
// it will move around normally.  A side effect of not being able to read the
// current state of the screen is that the cursor will erase anything it moves
// over while the left mouse button is not pressed.
//
// \return None.
//
//*****************************************************************************
unsigned short old_pixels[(DISPLAY_MOUSE_SIZE+1)][(DISPLAY_MOUSE_SIZE+1)];
unsigned int old_imageCount;

void
UpdateCursor(int iXDelta, int iYDelta)
{
    int iTemp, i, j;
    unsigned short * frame_buffer;

    frame_buffer = (unsigned short *)contextInfo[old_imageCount].pImageAddr;
	//
	// Erase the previous cursor.
	//
    for(i=0;i<(DISPLAY_MOUSE_SIZE+1);i++)
    	for(j=0;j<(DISPLAY_MOUSE_SIZE+1);j++)
    	{
    		*(frame_buffer + (((g_sCursor.sYMin+i)*WIDTH) + ((g_sCursor.sXMin+j)+2)) + 16 - 2) = old_pixels[i][j];
   	}


    //
    // Need to do signed math so use the temporary signed value.
    //
    iTemp = g_sCursor.sXMin;

    //
    // Update the X position without going off the screen.
    //
    if(((int)g_sCursor.sXMin + iXDelta + DISPLAY_MOUSE_SIZE) <
       GrContextDpyWidthGet(GetCurrentContext(imageCount)))
    {
        //
        // Update the X cursor position.
        //
        iTemp += iXDelta;

        //
        // Don't let the cursor go off the left of the screen either.
        //
        if(iTemp < 0)
        {
            iTemp = 0;
        }
    }

    //
    // Update the X position.
    //
    g_sCursor.sXMin = iTemp;
    g_sCursor.sXMax = iTemp + DISPLAY_MOUSE_SIZE;

    //
    // Need to do signed math so use the temporary signed value.
    //
    iTemp = g_sCursor.sYMin;

    //
    // Update the Y position without going off the screen.
    //
    if(((int)g_sCursor.sYMin + iYDelta) < (GrContextDpyHeightGet(GetCurrentContext(imageCount)) -
       DISPLAY_BANNER_HEIGHT - (DISPLAY_MOUSE_SIZE+1)))
    {
        //
        // Update the Y cursor position.
        //
        iTemp += iYDelta;

        //
        // Don't let the cursor overwrite the status area of the screen.
        //
        if(iTemp < DISPLAY_BANNER_HEIGHT + 1)
        {
            iTemp = DISPLAY_BANNER_HEIGHT + 1;
        }
    }

    //
    // Update the Y position.
    //
    g_sCursor.sYMin = iTemp;
    g_sCursor.sYMax = iTemp + DISPLAY_MOUSE_SIZE;

    // Save current pixel values that the cursor is about to overwrite
    old_imageCount = imageCount;
    for(i=0;i<(DISPLAY_MOUSE_SIZE+1);i++)
    	for(j=0;j<(DISPLAY_MOUSE_SIZE+1);j++)
    	{
    	    old_pixels[i][j] = contextInfo[imageCount].pImageAddr[(((g_sCursor.sYMin+i)*WIDTH) + ((g_sCursor.sXMin+j)+2)) + 16 - 2];
    	}

    //
    // Draw the new cursor.
    //
    GrContextForegroundSet(GetCurrentContext(imageCount), DISPLAY_MOUSE_FG);
    GrRectFill(GetCurrentContext(imageCount), &g_sCursor);
}

/*
** Enable all the peripherals in use
*/
static void PeripheralsSetUp(void)
{
    /* pin mux for I2C */
    I2CPinMuxSetup(0);

    /* PSC and pinmux for Raster */
    PSCModuleControl(SOC_PSC_1_REGS, HW_PSC_LCDC, PSC_POWERDOMAIN_ALWAYS_ON,
		     PSC_MDCTL_NEXT_ENABLE);
    LCDPinMuxSetup();

    /* PSC for McASP and EDMA, pinmux for McASP*/
    PSCModuleControl(SOC_PSC_1_REGS, HW_PSC_MCASP0, PSC_POWERDOMAIN_ALWAYS_ON,
		     PSC_MDCTL_NEXT_ENABLE);
    PSCModuleControl(SOC_PSC_0_REGS, HW_PSC_CC0, PSC_POWERDOMAIN_ALWAYS_ON,
		     PSC_MDCTL_NEXT_ENABLE);
    PSCModuleControl(SOC_PSC_0_REGS, HW_PSC_TC0, PSC_POWERDOMAIN_ALWAYS_ON,
		     PSC_MDCTL_NEXT_ENABLE);
    McASPPinMuxSetup();

    /* PSC and pinmux for GPIO */
    PSCModuleControl(SOC_PSC_1_REGS,HW_PSC_GPIO, PSC_POWERDOMAIN_ALWAYS_ON,
		     PSC_MDCTL_NEXT_ENABLE);
    GPIOBank6Pin12PinMuxSetup();

    /* PSC and pinmux for EMAC and MDIO */
    PSCModuleControl(SOC_PSC_1_REGS, HW_PSC_EMAC, PSC_POWERDOMAIN_ALWAYS_ON,
		     PSC_MDCTL_NEXT_ENABLE);
    EMACPinMuxSetup();

    /* Pinmux for RTC */
    RTCPinMuxSetup(FALSE);
}

/*
** This function clears any previous context info
*/
static void PrevContextClear(void)
{
    /* Close the previous state */
    Timer2Stop();
    tmrFlag  = FALSE;
    rtcSetFlag = FALSE;

    /* Make sure the LEDs are Off */
    LedOff();

}

/*
** Take the actions on the touch.
** The coordinates are given by the parameters
*/
static void CoOrdAction(int x, int y)
{
    int const *coOrd;
    unsigned int cnt;
    unsigned int numIcon = contextInfo[imageCount].numIcon;
    TOUCHSPEC const *touchSpec;

    for(cnt = 0; cnt < numIcon; cnt++)
    {
        touchSpec = contextInfo[imageCount].touchSpec + cnt;
        coOrd = touchSpec->coOrd;

        /* Take action only for proper coordinates */
        if((x >= coOrd[0]) && (x <= coOrd[1]) &&
           ((y >= coOrd[2]) && (y <= coOrd[3])))
        {
            PrevContextClear();            

			if((1 == cnt) && (1 != imageCount))
			{
				prevAction = 1;
			}
            (touchSpec->action)();
			break;
        }
    
    }         
}

/*
** Take the actions on click. 
*/
static void ClickAction(void)
{
    TOUCHSPEC const *clickSpec;

    /*
    ** Get the spec. Assumed that the last touch spec only will give 
    ** action for the next image.
    */
    clickSpec = contextInfo[clickIdx - 1].touchSpec;
                //+ (contextInfo[clickIdx - 1].numIcon - 1) ;

    PrevContextClear();

    (clickSpec->action)();
}

//*****************************************************************************
//
// This is the generic callback from host stack.
//
// \param pvData is actually a pointer to a tEventInfo structure.
//
// This function will be called to inform the application when a USB event has
// occured that is outside those releated to the mouse device.  At this
// point this is used to detect unsupported devices being inserted and removed.
// It is also used to inform the application when a power fault has occured.
// This function is required when the g_USBGenericEventDriver is included in
// the host controller driver array that is passed in to the
// USBHCDRegisterDrivers() function.
//
// \return None.
//
//*****************************************************************************
void
USBHCDEvents(void *pvData)
{
    tEventInfo *pEventInfo;

    //
    // Cast this pointer to its actual type.
    //
    pEventInfo = (tEventInfo *)pvData;

    switch(pEventInfo->ulEvent)
    {
        //
        // New mouse detected.
        //
        case USB_EVENT_CONNECTED:
        {
            //
            // An unknown device was detected.
            //
            eUSBState = STATE_UNKNOWN_DEVICE;

            break;
        }

        //
        // Keyboard has been unplugged.
        //
        case USB_EVENT_DISCONNECTED:
        {
            //
            // Unknown device has been removed.
            //
            eUSBState = STATE_NO_DEVICE;

            break;
        }

        default:
        {
            break;
        }
    }
}

//*****************************************************************************
//
// This is the callback from the USB HID mouse handler.
//
// \param pvCBData is ignored by this function.
// \param ulEvent is one of the valid events for a mouse device.
// \param ulMsgParam is defined by the event that occurs.
// \param pvMsgData is a pointer to data that is defined by the event that
// occurs.
//
// This function will be called to inform the application when a mouse has
// been plugged in or removed and anytime mouse movement or button pressed
// is detected.
//
// \return This function will return 0.
//
//*****************************************************************************
unsigned int
MouseCallback(void *pvCBData, unsigned int ulEvent, unsigned int ulMsgParam,
              void *pvMsgData)
{
    switch(ulEvent)
    {
        //
        // New mouse detected.
        //
        case USB_EVENT_CONNECTED:
        {

            //
            // Proceed to the STATE_MOUSE_INIT state so that the main loop can
            // finish initialized the mouse since USBHMouseInit() cannot be
            // called from within a callback.
            //
            eUSBState = STATE_MOUSE_INIT;

            break;
        }

        //
        // Mouse has been unplugged.
        //
        case USB_EVENT_DISCONNECTED:
        {
            //
            // Change the state so that the main loop knows that the mouse is
            // no longer present.
            //
            eUSBState = STATE_NO_DEVICE;

            //
            // Reset the button state.
            //
            g_ulButtons = 0;

            break;
        }

        //
        // Mouse button press detected.
        //
        case USBH_EVENT_HID_MS_PRESS:
        {
            //
            // Save the new button that was pressed.
            //
            g_ulButtons |= ulMsgParam;
            clicked = 1;

            break;
        }

        //
        // Mouse button release detected.
        //
        case USBH_EVENT_HID_MS_REL:
        {
            //
            // Remove the button from the pressed state.
            //
            g_ulButtons &= ~ulMsgParam;

            break;
        }

        //
        // Mouse X movement detected.
        //
        case USBH_EVENT_HID_MS_X:
        {
            //
            // Update the cursor on the screen.
            //
               UpdateCursor((signed char)ulMsgParam, 0);

            break;
        }

        //
        // Mouse Y movement detected.
        //
        case USBH_EVENT_HID_MS_Y:
        {
            //
            // Update the cursor on the screen.
            //
               UpdateCursor(0, (signed char)ulMsgParam);

            break;
        }
    }


    return(0);
}

/*
** Main function. The application starts here.
*/
int main(void)
{
    int x;
    int y;
    
    /* Initialize slide contents */
    InitializeSlides();

    PeripheralsSetUp();

    /* Initialize the Interrupt Controller */
#ifdef _TMS320C6X
    IntDSPINTCInit();
#else
    IntAINTCInit();
#endif

#ifdef _TMS320C6X
    /* Register the ISRs */  
    Raster0IntRegister(C674X_MASK_INT4);
    I2C0IntRegister(C674X_MASK_INT5);
    Timer2IntRegister(C674X_MASK_INT6);
    Gpio0IntRegister(C674X_MASK_INT7);
    EnetIntRegister(C674X_MASK_INT8, C674X_MASK_INT9);
    RtcIntRegister(C674X_MASK_INT12);
    USB0HostIntRegister(C674X_MASK_INT11);

    IntEnable(C674X_MASK_INT4);
    IntEnable(C674X_MASK_INT5);
    IntEnable(C674X_MASK_INT6);
    IntEnable(C674X_MASK_INT7);
    IntEnable(C674X_MASK_INT8);
    IntEnable(C674X_MASK_INT9);
    IntEnable(C674X_MASK_INT11);
    IntEnable(C674X_MASK_INT12);

    IntGlobalEnable();
#else
    Raster0IntRegister(2);
    I2C0IntRegister(4);
    Timer2IntRegister(3);
    Gpio0IntRegister(4);
    EnetIntRegister(2);
    RtcIntRegister(3);
    USB0HostIntRegister(2);

    IntMasterIRQEnable();

    /* Enable interrrupts at AINTC */
    IntGlobalEnable();
    IntIRQEnable();

    IntSystemEnable(SYS_INT_LCDINT); 
    IntSystemEnable(SYS_INT_I2CINT0); 
    IntSystemEnable(SYS_INT_TIMR2_ALL);
    IntSystemEnable(SYS_INT_GPIOB4);
    IntSystemEnable(SYS_INT_C0_RX);
    IntSystemEnable(SYS_INT_C0_TX);
    IntSystemEnable(SYS_INT_RTC);
    IntSystemEnable(SYS_INT_USB0);
#endif

    DelayTimerSetup();
    Raster0Init();
    AIC31Init();
    ToneLoopInit();
    RtcInit();
    UARTStdioInit(); 
    Timer2Config();
    Gpio0CardDetConfig();
    LedIfConfig();

    Raster0EOFIntEnable();
    Timer2IntEnable();
    Gpio0IntEnable();
    RtcSecIntEnable();

    imageCount = 0;
    Raster0Start(); 

    /* Start playing the tone */
    ToneLoopStart();

    //
    // Register the host class drivers.
    //
    USBHCDRegisterDrivers(0, g_ppHostClassDrivers, g_ulNumHostClassDrivers);

    //
    // Initialize the cursor.
    //
    g_ulButtons = 0;
    g_sCursor.sXMin = GrContextDpyWidthGet(GetCurrentContext(imageCount)) / 2;
    g_sCursor.sXMax = g_sCursor.sXMin + DISPLAY_MOUSE_SIZE;
    g_sCursor.sYMin = GrContextDpyHeightGet(GetCurrentContext(imageCount)) / 2;
    g_sCursor.sYMax = g_sCursor.sYMin + DISPLAY_MOUSE_SIZE;

    //
    //  Update the cursor once to display it.
    //
    UpdateCursor(0, 0);

    //
    // Open an instance of the mouse driver.  The mouse does not need
    // to be present at this time, this just saves a place for it and allows
    // the applications to be notified when a mouse is present.
    //
    g_ulMouseInstance =
        USBHMouseOpen(MouseCallback, g_pucMouseBuffer, MOUSE_MEMORY_SIZE);

    //
    // Initialize the power configuration. This sets the power enable signal
    // to be active high and does not enable the power fault.
    //
    USBHCDPowerConfigInit(0, USBHCD_VBUS_AUTO_HIGH);

    //
    // Initialize the host controller stack.
    //
    USBHCDInit(0, g_pHCDPool, HCD_MEMORY_SIZE);

    //
    // Call the main loop for the Host controller driver.
    //
    USBHCDMain();

	if(prevAction)
	{
		/* Go to prev image */
	    imageCount--;
	}
	else
	{
	    /* Go to next image */
	    (0 == clickIdx) ? imageCount++ : (imageCount = clickIdx); 
	}
	prevAction = 0;

    /*
    ** Loop for ever. Necessary actions shall be taken
    ** after detecting touch, based on the coordinates
    */
    while(1)
    {
        switch(eUSBState)
         {
             //
             // This state is entered when the mouse is first detected.
             //
             case STATE_MOUSE_INIT:
             {
                 //
                 // Initialize the newly connected mouse.
                 //
                 USBHMouseInit(g_ulMouseInstance);

                 //
                 // Proceed to the mouse connected state.
                 //
                 eUSBState = STATE_MOUSE_CONNECTED;

                 break;
             }
             case STATE_MOUSE_CONNECTED:
             {
                 //
                 // Nothing is currently done in the main loop when the mouse
                 // is connected.
                 //
                 break;
             }
             case STATE_NO_DEVICE:
             {
                 //
                 // The mouse is not connected so nothing needs to be done here.
                 //
                 break;
             }
             default:
             {
                 break;
             }
         }

         //
         // Periodically call the main loop for the Host controller driver.
         //
         USBHCDMain();

         // Check if touch is detected

         if(clicked == 1)
         {
        	 clicked = 0;

             // Validate the coordinates and take action

             CoOrdAction(g_sCursor.sXMin, g_sCursor.sYMin);
         }


         // Check if click is detected


         if(clickIdx != 0)
         {

             // Take the Action for click

             ClickAction();

             clickIdx = 0;
         }
        

         // Check if the Timer Expired

         if(TRUE == tmrFlag)
         {
             tmrFlag = FALSE;

             // Toggle the LED state
             LedToggle();
         }
 

         // Check if MMC/SD card is inserted

         if(TRUE == cardDetFlag)
         {
             delay(1);
             cardDetFlag = FALSE;
              
             if(TRUE == GpioCardInserted()) 
             {
                 UARTPuts("\n\rMMC/SD card inserted.\n\r", -1);
             }
        
             else
             {
                 UARTPuts("\n\rMMC/SD card removed.\n\r", -1);
             }
         }
 

         // Check if RTC Time is set

         if(TRUE == rtcSetFlag)
         {
             if(TRUE == rtcSecUpdate)
             { 
                 rtcSecUpdate = FALSE;
                 RtcTimeCalDisplay();
             }
         } 
    }
}

/*
** Action to be taken when the demo is to be driven via Ethernet
*/
static void ActionEnetInit(void)
{
    char ipMsg[60] = {"http://"};
    unsigned int i_index, i_msg = 7, ipByte = 0;
    unsigned int linkFlag = FALSE;
 

	if(prevAction)
	{
		/* Go to prev image */
	    imageCount--;
	}
	else
	{
	    /* Go to next image */
	    (0 == clickIdx) ? imageCount++ : (imageCount = clickIdx); 
	}
	prevAction = 0;
   
    /* restore  original slide contents */
    ResetChoiceSlide();
       
    GrContextForegroundSet(&sContextChoice, ClrRed);
    GrContextFontSet(&sContextChoice, &g_sFontCmss18b);
    GrStringDrawCentered(&sContextChoice, 
                         "Checking Ethernet link and acquiring IP address...", 
                         -1, 240, 235, 0);

    if(!EnetIfIsUp())
    {
        linkFlag = FALSE;
        EnetHttpServerInit();

        if(ipAddr)
        {
            linkFlag = TRUE;
        }
    }

    else
    {
        if(EnetLinkIsUp())
        {
            linkFlag = TRUE;
            UARTPuts("\n\rEVM OMAPL138 IP Address: ", -1);
            IpAddrDisplay();
        }
   
        else
        {
            linkFlag = FALSE;
        }
    }
 
    /* restore  original slide contents */
    ResetChoiceSlide();

    if((TRUE == linkFlag) && (ipAddr != 0))
    {
        for(i_index = 0; i_index < 4; i_index++)
        {
            ipByte = 0x000000FF & (ipAddr >> ((i_index)*8) );

            if(ipByte/100)
            {
                ipMsg[i_msg++] = (ipByte/100) + 48;
                ipByte = ipByte%100;
                ipMsg[i_msg++] = (ipByte/10) + 48;
                ipByte = ipByte%10;
                ipMsg[i_msg++] = ipByte + 48;
                ipByte = 0;
            }
            if(ipByte/10)
            {
                ipMsg[i_msg++] = (ipByte/10) + 48;
                ipByte = ipByte%10;
                ipMsg[i_msg++] = ipByte + 48;
                ipByte = 0;
            }
            if(ipByte)
            {
                ipMsg[i_msg++] = ipByte + 48;
            }

            ipMsg[i_msg++] = '.';
        }
        ipMsg[--i_msg] = '\0';

        strcat(ipMsg, "/index.html");
        GrContextForegroundSet(&sContextChoice, ClrRed);
        GrStringDrawCentered(&sContextChoice, ipMsg, -1, 240, 235, 0);
    }

    else
    {
        UARTPuts("\n\rNetwork Connection failed.\n\r", -1);
        
        GrStringDrawCentered(&sContextChoice, "Network Connectivity NOT Detected",
                             -1, 240, 235, 0);
    }
}

/*
** Action when no touch is detected 
*/
static void ActionIdle(void)
{
    ; /* Perform nothing */
}

/*
** Action for Menu
*/
static void ActionMenu(void)
{
	imageCount = 1;
	prevAction = 0;
}

/*
** Action for menu introduction icon click
*/
static void ActionMenuIntro(void) { 	imageCount++;}
/*
** Action for menu webdemo icon click
*/
static void ActionMenuWebDemo(void) { 	clickIdx = CLICK_IDX_CHOICE;}
/*
** Action for menu McASP icon click
*/
static void ActionMenuMcASP(void) { 	clickIdx = CLICK_IDX_MCASP;}

/*
** Action for menu Uart icon click
*/
static void ActionMenuUart(void) { 	clickIdx = CLICK_IDX_UART;}
/*
** Action for menu GPIO icon click
*/
static void ActionMenuGPIO(void) { 	clickIdx = CLICK_IDX_GPIO;}
/*
** Action for menu RTC icon click
*/
static void ActionMenuRTC(void) { 	clickIdx = CLICK_IDX_RTC;}

/*
** Action for menu USB icon click
*/
static void ActionMenuUSBMouse(void) { 	clickIdx = CLICK_IDX_USB_MOUSE;}
/*
** Action for menu timer icon click
*/
static void ActionMenuTimer(void) { 	clickIdx = CLICK_IDX_TIMER;}
/*
** Action for menu ethernet icon click
*/
static void ActionMenuEthernet(void) { 	clickIdx = CLICK_IDX_ETHERNET;}


/*
** Default action 
*/
static void ActionDefault(void)
{
	if(prevAction)
	{
		/* Go to prev image */
	    imageCount--;
	}
	else
	{
	    (0 == clickIdx) ? imageCount++ : (imageCount = clickIdx); 
	}
	prevAction = 0;
}

/*
** Action for LED blink through I2C
*/
static void ActionI2C(void)
{
    /* Display the proper Image */ 
	if(prevAction)
	{
		/* Go to prev image */
	    imageCount--;
	}
	else
	{
	    (0 == clickIdx) ? imageCount++ : (imageCount = clickIdx); 
	}
	prevAction = 0;

    tmrFlag  = FALSE;
    tmrStepVary = FALSE;
  
    /* Start the timer */
    Timer2Start();
}

/*
** Action for LED blink with varying period
*/
static void ActionTimer(void)
{
    /* Display the proper Image */ 
	if(prevAction)
	{
		/* Go to prev image */
	    imageCount--;
	}
	else
	{
	    (0 == clickIdx) ? imageCount++ : (imageCount = clickIdx); 
	}
	prevAction = 0;    
  
    tmrFlag  = FALSE;
    tmrStepVary = TRUE;

    Timer2Start();
}

/*
** Action for GPIO card detection
*/
static void ActionGpio(void)
{
    /* Display the proper Image */ 
	if(prevAction)
	{
		/* Go to prev image */
	    imageCount--;
	}
	else
	{
	    (0 == clickIdx) ? imageCount++ : (imageCount = clickIdx); 
	}
	prevAction = 0;
}

static void ActionUSBMouse(void)
{
	/* Display the proper Image */ 
	if(prevAction)
	{
		/* Go to prev image */
	    imageCount--;
	}
	else
	{
	    (0 == clickIdx) ? imageCount++ : (imageCount = clickIdx); 
	}
	prevAction = 0;
	
}




/*
** Action for RTC Time Set. This is a blocking call.
*/
static void ActionTimeSet(void)
{
    RtcTimeCalSet(); 
}


/*
** Delay Generator
*/
//static void Delay(volatile unsigned int delay)
//{
//    while(delay--);
//}

/****************************** End of file *********************************/
  
