/**
 *  \file       notify.h
 *
 *  \brief      Notification driver for no-OS IPC
 *
 *  The Notify module manages the signalling of events from one processor to 
 *  another in a multiprocessor system. The notification mechanism performs
 *  the handshaking of software interrupts over IPC hardware interrupts.
 *  The model supports both polling and enabling methods for interrupts
 *  in a non-OS run-time environment.
 *
 *  In order to use any Notify APIs, IPC must first be started with the 
 *  "Ipc_start" call.  This internally calls #Notify_attach which sets up
 *  all necessary Notify drivers, data structures and interprocessor interrupts.
 * 
 *  To receive notifications, a processor registers to an eventId using
 *  #Notify_registerEvent or #Notify_registerEventSingle. These two APIs (like
 *  most other Notify APIs) uses a MultiProc id and line id to target a specific 
 *    interrupt line to/from a specific processor on a device. A callback
 *    function can be included to be executed when the event is triggered. Note
 *  that the callback function will be called by the ISR. Because of this, it 
 *    should be a very short function to keep latency overhead low.
 *
 *  Once an event has been registered, a remote processor may send an event
 *  using the #Notify_sendEvent call.  On the receiving (local) side, if the 
 *  event and the interrupt line are both enabled, the callback function
 *  registered to the event will be called by the Notify mechanism.
 *
 *  The application can wait for an event using #Notify_waitEvent. This API acts
 *    as a single read access to the event table. Regardless whether interrupt or 
 *  polling is used, the API will block until an event is triggered. 
 *
 *  A specific event may be disabled or enabled using the #Notify_disableEvent
 *  and #Notify_enableEvent calls.  An entire interrupt line may be disabled
 *  or restored using the #Notify_disable or #Notify_restore calls.  The
 *  Notify_disable call does not alter the state of individual events.
 *  Instead, it just disables the ability of the Notify module to receive
 *  events on the interrupt line.
 *
 *  Notify APIs should never be called within an ISR context.  All API calls 
 *  should be made within main() or a thread with the exception of 
 *  Notify_sendEvent which may also be called from within an ISR.
 *
 *  The Notify header should be included in an application as follows:
 *  \code
 *  #include "ipclite/notify.h"
 *  \endcode
 *
 *  \version  0.00.01
 */

/* Copyright (c) 2011, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */


#ifndef NOTIFY_H_
#define NOTIFY_H_

#if defined (__cplusplus)
extern "C" {
#endif

#include    "ipc.h"

/******************************************************************************
**              All success and failure codes for the module
******************************************************************************/

/**
 *  \def    Notify_S_BUSY
 *  \brief  The resource is still in use  
 */
#define Notify_S_BUSY                    2
 
/**
 *  \def    Notify_S_ALREADYSETUP
 *  \brief  Module already set up
 */
#define Notify_S_ALREADYSETUP            1

/**
 *  \def    Notify_S_SUCCESS
 *  \brief  Operation is successful.
 */
#define Notify_S_SUCCESS                 0

/**
 *  \def    Notify_E_FAIL
 *  \brief  Generic failure.
 */
#define Notify_E_FAIL                   -1

/**
 *  \def    Notify_E_INVALIDARG
 *  \brief  Argument passed to function is invalid.
 */
#define Notify_E_INVALIDARG             -2

/**
 *  \def    Notify_E_MEMORY
 *  \brief  Operation resulted in memory failure.
 */
#define Notify_E_MEMORY                 -3

/**
 *  \def    Notify_E_ALREADYEXISTS
 *  \brief  The specified entity already exists.
 */
#define Notify_E_ALREADYEXISTS          -4

/**
 *  \def    Notify_E_NOTFOUND
 *  \brief  Unable to find the specified entity.
 */
#define Notify_E_NOTFOUND               -5

/**
 *  \def    Notify_E_TIMEOUT
 *  \brief  Operation timed out.
 */
#define Notify_E_TIMEOUT                -6

/**
 *  \def    Notify_E_INVALIDSTATE
 *  \brief  Module is not initialized.
 */
#define Notify_E_INVALIDSTATE           -7

/**
 *  \def    Notify_E_OSFAILURE
 *  \brief  A failure occurred in an OS-specific call  
 */
#define Notify_E_OSFAILURE              -8

/**
 *  \def    Notify_E_ALREADYSETUP
 *  \brief  The module has been already setup 
 */
#define Notify_E_ALREADYSETUP           -9

/**
 *  \def    Notify_E_RESOURCE
 *  \brief  Specified resource is not available  
 */
#define Notify_E_RESOURCE               -10

/**
 *  \def    Notify_E_RESTART
 *  \brief  Operation was interrupted. Please restart the operation  
 */
#define Notify_E_RESTART                -11

/**
 *  \def    Notify_E_BUSY
 *  \brief  The resource is still in use  
 */
#define Notify_E_BUSY                   -12

/**
 *  \def    Notify_E_DRIVERNOTREGISTERED
 *  \brief  Driver corresponding to the specified eventId is not registered
 */
#define Notify_E_DRIVERNOTREGISTERED    -13

/**
 *  \def    Notify_E_EVTNOTREGISTERED
 *  \brief  Event not registered
 */
#define Notify_E_EVTNOTREGISTERED       -14

/**
 *  \def    Notify_E_EVTDISABLED
 *  \brief  Event is disabled
 */
#define Notify_E_EVTDISABLED            -15

/**
 *  \def    Notify_E_NOTINITIALIZED
 *  \brief  Remote notification is not initialized
 */
#define Notify_E_NOTINITIALIZED         -16

/**
 *  \def    Notify_E_EVTRESERVED
 *  \brief  Trying to illegally use a reserved event
 */
#define Notify_E_EVTRESERVED            -17

/**
 *     @def    Notify_E_EVTOVERRUN
 *     @brief    Previous event has not been cleared by remote processor callback
 */
#define    Notify_E_EVTOVERRUN          -18

/******************************************************************************
**                          Defines & Macros
******************************************************************************/

/**
 *  \def    Notify_MAXEVENTS
 *  \brief  Maximum number of events supported by the Notify module. It is
 *          arbitrary and can be increased if memory is available.
 */
#define Notify_MAXEVENTS                (UInt16)16

/**
 *  \def    Notify_MAX_INTLINES
 *  \brief  Maximum number of interrupt lines for any pair of processors
 */
#define Notify_MAX_INTLINES             2


/******************************************************************************
**                          Structures & Enums
******************************************************************************/

/**
 *  \var     Notify_FnNotifyCbck
 *  \brief   Signature of any callback function that can be registered with the 
 *           Notify component.
 *
 *  \param[in]  procId      Remote processor id
 *  \param[in]  lineId      Line id
 *  \param[in]  eventId     Event id (minus system key if reserved event)
 *  \param[in]  arg         Argument specified in the registerEvent
 *  \param[in]  payload     Payload specified in the sendEvent      
 */
typedef void (*Notify_FnNotifyCbck)(UInt16 , UInt16, UInt32, UArg, UInt32);

/******************************************************************************
**                      Notify Module-wide Functions
******************************************************************************/

 /**
  *  \brief     Start Notify mechanism after synchronizing with remote processor.
  *              IPC Interrupt will be enabled if IPC_INTERRUPT_METHOD was selected
  *
  *  \return     Notify status:
  *              - #Notify_E_FAIL: failed to connect to remote processor
  *              - #Notify_S_SUCCESS: successfully start
  */
 Int    Notify_start(void);

/**
 *  \brief      Disable interrupt-driven notification from remote processor
 *
 *  This function disables the IPC interrupt from remote CPU. This means the local
 *    processor will ignore interrupt-driven event notifications from the remote processor. To
 *    process a pending event while in this mode, use the Notify_waitEvent API.
 *    While there's a pending event, remote Notify_sendEvent to the same eventId will be blocked
 *    until this event is processsed.
 *
 *  \param[in]  procId      Remote processor id
 *  \params[in]    lineId            "  interrupt line id
 *
 *  \return     Notify status:
 *              - #Notify_E_INVALIDARG: invalid procId or lineId value
 *              - #Notify_E_FAIL: failed to connect to remote processor
 *              - #Notify_S_SUCCESS: successfully restore Notify
 */
Int Notify_disable(UInt16 procId, UInt16 lineId);

/**
 *  \brief      Restore Notify process
 *
 *  This function restores the Notify mechanism that was disabled.
 *  If interrupt method was selected, ipc interrupt will be re-enabled to
 *  receive notification. Otherwise there is no change in state, i.e. polling
 *  method needs be used to receive notification via the Notify_waitEvent API.
 *
 *  \param[in]  procId      Remote processor id
 *  \params[in]    lineId        "       interrupt line id
 *
 *  \return     Notify status:
 *              - #Notify_E_INVALIDARG: invalid procId or lineId value
 *              - #Notify_E_FAIL: failed to connect to remote processor
 *              - #Notify_S_SUCCESS: successfully restore Notify
 */
Int Notify_restore(UInt16 procId, UInt16 lineId);

/**
 *  \brief      Send an event on an interrupt line
 *
 *  This function sends an event to a processor via an interrupt line
 *  identified by a processor id and line id.
 *
 *  On the destination processor, the callback function registered
 *  with Notify with the associated eventId and source processor id
 *  will be called.
 *
 *  The 'waitClear' flag, if set, will block the call until the event has
 *  been acknowledged by the remote processor. If the remote CPU is using
 *  polling notification method to receive, this API call will be blocked
 *  until the event is processed.
 *  If there's already an unprocessed event previously sent to the same eventId
 *  the call will return with fail status.
 *
 *  Notify_sendEvent can be called from a hardware interrupt context
 *  unlike other Notify APIs if waitClear is set false, but if the remote
 *  CPU disables interrupts, the event delivery cannot be guaranteed.
 *
 *  \param[in]  procId      Remote processor id
 *  \param[in]  lineId      Line id
 *  \param[in]  eventId     Event id
 *  \param[in]  payload     Payload to be sent along with the event.
 *  \param[in]  waitClear   Indicates whether Notify driver will wait for
 *                          previous event to be cleared. If a payload needs to
 *                          be sent across, this must be TRUE.
 *  
 *  \return     Notify status:
 *              - #Notify_E_INVALIDARG: invalid procId or lineId value
 *              - #Notify_E_EVTNOTREGISTERED: event has no registered callback 
 *                functions
 *              - #Notify_E_NOTINITIALIZED: remote driver has not yet been
 *                initialized
 *              - #Notify_E_EVTDISABLED: remote event is disabled
 *              - #Notify_E_BUSY: unprocessed event with this eventId on remote side
 *              - #Notify_E_TIMEOUT: timeout occured (when waitClear is TRUE)
 *              - #Notify_S_SUCCESS: event successfully sent
 */
Int Notify_sendEvent(UInt16 procId, 
                     UInt16 lineId,
                     UInt32 eventId, 
                     UInt32 payload,
                     Bool waitClear);
                     
/**
 *    @brief        Wait for any event
 *
 *    This function checks for any event pending on interrupt line of a remote
 *    processor Id, both passed as arguments.
 *
 *    If there's an event pending, it will process it by executing the
 *    call back function registered with that event.
 *    This API is needed for the Ipc_POLLING method where the only way to get
 *  to the event from the application in a non-OS run-time environment.
 *
 *  \param[in]  procId      Remote processor id
 *  \param[in]  lineId      Line id
 * 
 *  \return     Notify status:
 *              - #Notify_E_FAIL: no event available from remote processor
 *              - #Notify_E_EVTNOTREGISTERED: event has no registered callback function
 *              - #Notify_S_SUCCESS: successfully attach to remote processor
 */
Int Notify_waitEvent(UInt16 procId, UInt16 lineId);

/**
 *  \brief      Disable an event
 *
 *  This function allows the disabling of a single event number
 *  associated with the specified remote processor and interrupt line.
 *  Sending to a disabled event will return #Notify_E_EVTDISABLED on the
 *  sender if waitClear is false.
 *
 *  An event is, by default, enabled upon driver initialization.
 *  Calling #Notify_disableEvent upon an event that is already disabled
 *  results in no change in state.
 *
 *  Note that callbacks may be registered to an event or removed
 *  from the event even while the event has been disabled.
 *
 *  \param[in]  procId      Remote processor id
 *  \param[in]  lineId      Line id
 *  \param[in]  eventId     Event id
 *
 *  \sa         Notify_enableEvent
 */
Void Notify_disableEvent(UInt16 procId, UInt16 lineId, UInt32 eventId); 

/**
 *  \brief      Enable an event
 *
 *  This function re-enables an event that has been previously disabled
 *  using disableEvent().  Calling Notify_enableEvent upon an event that is
 *  already enabled results in no change in state.  An event is,
 *  by default, enabled upon IPC module initialization.
 *
 *  \param[in]  procId      Remote processor id
 *  \param[in]  lineId      Line id
 *  \param[in]  eventId     Event id
 *
 *  \sa         Notify_disableEvent
 */
Void Notify_enableEvent(UInt16 procId, UInt16 lineId, UInt32 eventId);

/**
 *  \brief      Register an event
 *
 *  This function registers a callback, processor id and interrupt line
 *  to a specific event number. When the event is received by the
 *  specified processor, the callback is called.
 *
 *  The callback function prototype is of type #Notify_FnNotifyCbck.
 *  This function must be non-blocking.
 *
 *  It is important to note that interrupts are disabled during the entire
 *  duration of this function's execution.
 *
 *  \param[in]  procId          Remote processor id
 *  \param[in]  lineId          Line id
 *  \param[in]  eventId         Event id
 *  \param[in]  fnNotifyCbck    Pointer to callback function
 *  \param[in]  cbckArg         Callback function argument
 *
 *  \return     Notify status:
 *              - #Notify_E_INVALIDARG: invalid procId or lineId value
 *              - #Notify_E_FAIL: cannot perform action - event already registered
 *              - #Notify_S_SUCCESS: Event successfully registered
 *
 *  \sa         Notify_unregisterEvent
 */
Int Notify_registerEvent(UInt16 procId, 
                         UInt16 lineId, 
                         UInt32 eventId,
                         Notify_FnNotifyCbck fnNotifyCbck,
                         UArg cbckArg);
       
/**
 *  \brief      Remove an event from event list
 *
 *  This function unregisters a single callback that was registered to an event
 *  using Notify_registerEvent. The procId, lineId, eventId, fnNotifyCbck and
 *  cbckArg must exactly match with the registered one.
 *
 *    The event number is marked available after this operation.
 *
 *  \param[in]  procId          Remote processor id
 *  \param[in]  lineId          Line id
 *  \param[in]  eventId         Event id
 *  \param[in]  fnNotifyCbck    Pointer to callback function
 *  \param[in]  cbckArg         Callback function argument
 *
 *  \return     Notify status:
 *              - #Notify_E_NOTFOUND: event listener not found
 *              - #Notify_E_INVALIDARG: invalid procId or lineId value
 *              - #Notify_S_SUCCESS: event listener unregistered
 *
 *  \sa         Notify_registerEvent
 */
Int Notify_unregisterEvent(UInt16 procId,
                        UInt16 lineId,
                        UInt32 eventId,
                        Notify_FnNotifyCbck fnNotifyCbck,
                        UArg cbckArg);


#if defined (__cplusplus)
}
#endif /* defined (__cplusplus) */
#endif /* NOTIFY_H_ */
