/**
 * \file  dsp_exception.c
 *
 * \brief Enable and test DSP exceptions.
 */

/*
* Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com/ 
*
*  Redistribution and use in source and binary forms, with or without 
*  modification, are permitted provided that the following conditions 
*  are met:
*
*    Redistributions of source code must retain the above copyright 
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the 
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/* Note: this inclusion allows us to access C6000 core registers more easily. */
#include <c6x.h>

#include "hw_types.h"
#include "soc_OMAPL138.h"
#include "lcdkOMAPL138.h"
#include "interrupt.h"
#include "hw_dspintc.h"

#include "timer.h"
#include "uartStdio.h"

#ifndef _TMS320C6X
#error This application is intended to be built for C6000 DSP only.
#endif

/******************************************************************************
**                      INTERNAL MACRO DEFINITIONS
*******************************************************************************/
#define TMR_PERIOD_LSB32               (0x07FFFFFF)
#define TMR_PERIOD_MSB32               (0x0)

/******************************************************************************
**                      INTERNAL FUNCTION PROTOTYPES
*******************************************************************************/

static void TimerSetUp64Bit(void);
static void exception_isr(void);

/******************************************************************************
**                      INTERNAL VARIABLE DEFINITIONS
*******************************************************************************/

volatile int isr_counter = 0;

/******************************************************************************
**                      INTERNAL FUNCTION DEFINITIONS
*******************************************************************************/

int main(void)
{
    int i;

    /* Enable the UART console */
    UARTStdioInit();
    UARTprintf("\n\nDSP Exception Example Application\n\n");

    /* Initialize interrupts */
    IntDSPINTCInit();
    IntGlobalEnable();
    IntRegister(C674X_MASK_EXC, exception_isr);
    UARTprintf("Interrupts enabled!\n");

    for (i = 1; i <= 10; i++)
    {
        /* Attempt to generate an exception */
        asm(" swe");
        UARTprintf("  Trying to generate a software exception...\n", i);

        /* Report current isr counter */
        UARTprintf("  Exception handler called %u times so far\n", isr_counter);
        
        /* Enable DSP exceptions halfway through */
        if (i == 5)
        {
            ExcGlobalEnable();
            UARTprintf("Exceptions enabled!\n");
        }
    }
    
    /* Use the Timer2 peripheral to generate more exceptions */
    TimerSetUp64Bit();
    ExcCombineAdd(SYS_INT_T64P2_TINTALL);
    UARTprintf("Generating more exceptions using the timer...\n");
    
    /* Start the timer */
    TimerEnable(SOC_TMR_2_REGS, TMR_TIMER12, TMR_ENABLE_CONT);
    
    i = isr_counter;
    while (isr_counter <= 10)
    {
        if (i != isr_counter)
        {
            i = isr_counter;
            UARTprintf("  Caught exception %u (generated by timer event)\n", i);

            /* Re-enable timer interrupt */
            TimerIntEnable(SOC_TMR_2_REGS, TMR_INT_TMR12_NON_CAPT_MODE);
        }
    }
    
    /* Disable the timer to stop generating exceptions */
    TimerDisable(SOC_TMR_2_REGS, TMR_TIMER12);
    
    UARTprintf("\nEnd of DSP Exception Example Application\n");

    while(1);
}

/*
** Setup the timer for 64 bit mode
*/
static void TimerSetUp64Bit(void)
{
    /* Configuration of Timer */
    TimerConfigure(SOC_TMR_2_REGS, TMR_CFG_64BIT_CLK_INT);

    /* Set the 64 bit timer period */
    TimerPeriodSet(SOC_TMR_2_REGS, TMR_TIMER12, TMR_PERIOD_LSB32);
    TimerPeriodSet(SOC_TMR_2_REGS, TMR_TIMER34, TMR_PERIOD_MSB32);
    
    /* Enable timer interrupt */
    TimerIntEnable(SOC_TMR_2_REGS, TMR_INT_TMR12_NON_CAPT_MODE);
}

/**
 * \brief   Interrupt Service Routine(ISR) to handle DSP exceptions.
 *          Increments a counter to track exceptions triggered by main().
 *
 * Ordinarily, you shouldn't try to return from an exception. It's generally a
 * better idea to use while(1) or similar to hang execution, then check the
 * value of IERR to determine what has gone wrong. This application generates
 * software exceptions intentionally, so we will bend the rules to continue
 * execution in main().
 */

static void exception_isr(void)
{
    isr_counter++;
    
    /* Note: for internal exceptions, this is the correct place to check the
     * IERR register to see what has gone wrong. In this case we generated
     * a software exception intentionally, so that step isn't necessary for
     * this application. */
    
    /* Clear exceptions */
    ECR = ~0u;
    
    /* Clear masked exceptions */
    HWREG(SOC_INTC_0_REGS + DSPINTC_EVTCLR(0)) = HWREG(SOC_INTC_0_REGS + DSPINTC_MEXPFLAG(0));

    /* Note: The NMI/exception handler in StarterWare is defined with the
     * "interrupt" keyword, which means it will try to return to the pointer
     * stored in IRP. Exceptions need to return to NRP instead, so we can
     * "trick" the ISR into branching to the correct return location by copying
     * NRP into IRP. This would cause problems if the exception occurred while
     * an interrupt was already being processed. */
    IRP = NRP;

    /* Use NTSR value for ITSR (see note above) */
    ITSR = NTSR;
}

/****************************END OF FILE*************************************/
