/**
 * \file spi.c
 *
 * \brief  This is a sample application file which invokes some APIs
 *         from the SPI device abstraction layer to perform configuration,
 *         transmission and reception operations.
 */

/*
* Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com/ 
*
*  Redistribution and use in source and binary forms, with or without 
*  modification, are permitted provided that the following conditions 
*  are met:
*
*    Redistributions of source code must retain the above copyright 
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the 
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <string.h>
#include "soc_OMAPL138.h"
#include "hw_psc_OMAPL138.h"
#include "lcdkOMAPL138.h"
#include "uart.h"
#include "spi.h"
#include "psc.h"
#include "interrupt.h"
#include "uartStdio.h"

/******************************************************************************
**                      INTERNAL MACRO DEFINITIONS
*******************************************************************************/
/* value to configure SMIO,SOMI,CLK and CS pin as functional pin */
#define SIMO_SOMI_CLK_CS        0x00000E04
#define CHAR_LENGTH             0x8

/******************************************************************************
**                      INTERNAL FUNCTION PROTOTYPES
*******************************************************************************/
static void SPIConfigDataFmtReg(unsigned int dataFormat);
static void SpiTransfer(void);
static void SetUpInt(void);
static void SetUpSPI(void);
static void GetStatusCommand(void);
static void ResetCommand(void);
void SPIIsr(void);

/******************************************************************************
**                      INTERNAL VARIABLE DEFINITIONS
*******************************************************************************/
volatile unsigned int flag = 1;
unsigned int tx_len;
unsigned int rx_len;
unsigned char vrf_data[260];
unsigned char tx_data[260];
volatile unsigned char rx_data[260];
unsigned char *p_tx;
volatile unsigned char *p_rx;
volatile unsigned char StatusResponseMessage[16];

/******************************************************************************
**                      INTERNAL FUNCTION DEFINITIONS
*******************************************************************************/
int main(void)
{
    /* Waking up the SPI1 instance. */
    PSCModuleControl(SOC_PSC_1_REGS, HW_PSC_SPI1, PSC_POWERDOMAIN_ALWAYS_ON,
                     PSC_MDCTL_NEXT_ENABLE);

    /* Initializing the UART instance for serial communication. */
    UARTStdioInit();

    UARTPuts("Welcome to StarterWare SPI application.\r\n\r\n", -1);
    UARTPuts("Here the SPI controller on the SoC communicates with", -1);
    UARTPuts(" the Fingerprint Sensor present on the LCDK.\r\n\r\n", -1);

    /* Performing the Pin Multiplexing for SPI1. */
    SPIPinMuxSetup(1);

    /* 
    ** Using the Chip Select(CS) 0 pin of SPI1 to communicate with the Fingerprint Sensor.
    */
    SPI1CSPinMuxSetup(2);

    /* Enable use of SPI1 interrupts. */
    SetUpInt();

    /* Configuring and enabling the SPI1 instance. */
    SetUpSPI();

    /* Issue a Reset to the Fingerprint Sensor */
    ResetCommand();
    UARTPuts("Attempting to Reset the Fingerprint Sensor.\r\n\r\n", -1);

    /* Read the Status of the Fingerprint Sensor */
    GetStatusCommand();
    if(StatusResponseMessage[11] == 0x48)
    {
        UARTPuts("Fingerprint Sensor Successfully Reset.\r\n\r\n", -1);
    }
    else
    {
        UARTPuts("ERROR: Fingerprint Sensor Reset Attempt Unsuccessful.\r\n\r\n", -1);
    }
}

static void GetStatusCommand(void)
{
	unsigned int i;

	// Issue Get Status command and read Status Message Response
	tx_data[0] = 0x80;
	tx_data[1] = 0x10;
	tx_data[2] = 0xBA;
	tx_data[3] = 0x07;
	tx_data[4] = 0x00;
	tx_data[5] = 0x00;
	tx_data[6] = 0x00;
	tx_data[7] = 0xFF;
	tx_data[8] = 0xFF;
	tx_data[9] = 0xFF;
	tx_data[10] = 0xFF;
	tx_data[11] = 0xFF;
	tx_data[12] = 0xFF;
	tx_data[13] = 0xFF;
	tx_data[14] = 0xFF;
	tx_data[15] = 0xFF;
	tx_data[16] = 0xFF;
	tx_data[17] = 0xFF;
	tx_data[18] = 0xFF;
	tx_data[19] = 0xFF;
	tx_data[20] = 0xFF;
	tx_data[21] = 0xFF;
	tx_data[22] = 0xFF;
	tx_len = 23;
	rx_len = 23;

	SPIDat1Config(SOC_SPI_1_REGS, (SPI_CSHOLD | SPI_DATA_FORMAT0), 0x4);
    SpiTransfer();

    for(i=0;i<16;i++)
    {
    	StatusResponseMessage[i] = rx_data[i+7];
    }
}

static void ResetCommand(void)
{
	// Issue Reset Command
	tx_data[0] = 0x80;
	tx_data[1] = 0x14;
	tx_data[2] = 0xBA;
	tx_data[3] = 0x07;
	tx_data[4] = 0x00;
	tx_data[5] = 0x00;
	tx_data[6] = 0x00;

	tx_len = 7;
	rx_len = 7;

	SPIDat1Config(SOC_SPI_1_REGS, (SPI_CSHOLD | SPI_DATA_FORMAT0), 0x4);
    SpiTransfer();
}

/*
** Configures ARM interrupt controller to generate SPI interrupt
**
*/
static void SetUpInt(void)
{
	// Setup the ARM or DSP interrupt controller

#ifdef _TMS320C6X
	// Initialize the DSP interrupt controller
	IntDSPINTCInit();

	// Register the ISR in the vector table
	IntRegister(C674X_MASK_INT4, SPIIsr);

	// Map system interrupt to the DSP maskable interrupt
	IntEventMap(C674X_MASK_INT4, SYS_INT_SPI1_INT);

	// Enable the DSP maskable interrupt
	IntEnable(C674X_MASK_INT4);

	// Enable DSP interrupts globally
	IntGlobalEnable();
#else
    /* Initialize the ARM Interrupt Controller.*/
    IntAINTCInit();

    /* Register the ISR in the Interrupt Vector Table.*/
    IntRegister(SYS_INT_SPINT1, SPIIsr);

    /* Set the channnel number 2 of AINTC for system interrupt 56.
     * Channel 2 is mapped to IRQ interrupt of ARM9.
    */
    IntChannelSet(SYS_INT_SPINT1, 2);

    /* Enable the System Interrupts for AINTC.*/
    IntSystemEnable(SYS_INT_SPINT1);

    /* Enable IRQ in CPSR.*/
    IntMasterIRQEnable();

    /* Enable the interrupts in GER of AINTC.*/
    IntGlobalEnable();

    /* Enable the interrupts in HIER of AINTC.*/
    IntIRQEnable();
#endif
}

/*
** Configures SPI Controller
**
*/
static void SetUpSPI(void)
{
    unsigned char cs  = 0x04;
    unsigned char dcs = 0x04;
    unsigned int  val = SIMO_SOMI_CLK_CS;
    SPIReset(SOC_SPI_1_REGS);

    SPIOutOfReset(SOC_SPI_1_REGS);

    SPIModeConfigure(SOC_SPI_1_REGS, SPI_MASTER_MODE);

    //SPIClkConfigure(SOC_SPI_1_REGS, 150000000, 20000000, SPI_DATA_FORMAT0);
    SPIClkConfigure(SOC_SPI_1_REGS, 150000000, 1000000, SPI_DATA_FORMAT0);

    SPIPinControl(SOC_SPI_1_REGS, 0, 0, &val);

    SPIDefaultCSSet(SOC_SPI_1_REGS, dcs);

    /* Configures SPI Data Format Register */
    SPIConfigDataFmtReg(SPI_DATA_FORMAT0);
  
     /* Selects the SPI Data format register to used and Sets CSHOLD 
      * to assert CS pin(line)  
      */
    SPIDat1Config(SOC_SPI_1_REGS, (SPI_CSHOLD | SPI_DATA_FORMAT0), cs);

     /* map interrupts to interrupt line INT1 */
    SPIIntLevelSet(SOC_SPI_1_REGS, SPI_RECV_INTLVL | SPI_TRANSMIT_INTLVL);
    
    /* Enable SPI communication */
    SPIEnable(SOC_SPI_1_REGS);
}
/*
** Configures Data Format register of SPI
**
*/
static void SPIConfigDataFmtReg(unsigned int dataFormat)
{
    /* Configures the polarity and phase of SPI clock */
    /*
	SPIConfigClkFormat(SOC_SPI_1_REGS,
                       (SPI_CLK_POL_HIGH | SPI_CLK_INPHASE),
                       dataFormat);
    */
    SPIConfigClkFormat(SOC_SPI_1_REGS,
                       (SPI_CLK_POL_LOW | SPI_CLK_INPHASE),
                       dataFormat);

    /* Configures SPI to transmit MSB bit First during data transfer */
    SPIShiftMsbFirst(SOC_SPI_1_REGS, dataFormat);

    /* Sets the Charcter length */
    SPICharLengthSet(SOC_SPI_1_REGS, CHAR_LENGTH, dataFormat);
}

/*
** Enables SPI Transmit and Receive interrupt.
** Deasserts Chip Select line.
*/
static void  SpiTransfer(void)
{
    p_tx = &tx_data[0];
    p_rx = &rx_data[0];
    SPIIntEnable(SOC_SPI_1_REGS, (SPI_RECV_INT | SPI_TRANSMIT_INT));
    while(flag);
    flag = 1;
    /* Deasserts the CS pin(line) */
    SPIDat1Config(SOC_SPI_1_REGS, SPI_DATA_FORMAT0, 0x4);
}

/*
** Data transmission and receiption SPIIsr
**
*/
void SPIIsr(void)
{
    unsigned int intCode = 0;

#ifdef _TMS320C6X
    IntEventClear(SYS_INT_SPI1_INT);
#else
    IntSystemStatusClear(56);
#endif

    intCode = SPIInterruptVectorGet(SOC_SPI_1_REGS);

    while (intCode)
    {
        if(intCode == SPI_TX_BUF_EMPTY)
        {
            tx_len--;
            SPITransmitData1(SOC_SPI_1_REGS, *p_tx);
            p_tx++;
            if (!tx_len)
            {
                SPIIntDisable(SOC_SPI_1_REGS, SPI_TRANSMIT_INT);
            }
        }

        if(intCode == SPI_RECV_FULL)
        {
            rx_len--;
            *p_rx = (char)SPIDataReceive(SOC_SPI_1_REGS);
            p_rx++;
            if (!rx_len)
            {
                flag = 0;
                SPIIntDisable(SOC_SPI_1_REGS, SPI_RECV_INT);
            }
        }

        intCode = SPIInterruptVectorGet(SOC_SPI_1_REGS);
    }
}

/******************************* End of file *********************************/
