/**
 * \file spi.c
 *
 * \brief  This is a sample application file which invokes some APIs
 *         from the SPI device abstraction layer to perform configuration,
 *         transmission and reception operations.
 */

/*
* Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com/ 
*
*  Redistribution and use in source and binary forms, with or without 
*  modification, are permitted provided that the following conditions 
*  are met:
*
*    Redistributions of source code must retain the above copyright 
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the 
*    documentation and/or other materials provided with the   
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "spi.h"
#include "psc.h"
#include "uart.h"
#include "edma.h"
#include "string.h"
#include "hw_types.h"
#include "interrupt.h"
#include "edma_event.h"
#include "lcdkOMAPL138.h"
#include "uartStdio.h"
#include "soc_OMAPL138.h"
#include "hw_psc_OMAPL138.h"
/******************************************************************************
**                      INTERNAL MACRO DEFINITIONS
*******************************************************************************/
/* value to configure SMIO,SOMI,CLK and CS pin as functional pin */
#define SIMO_SOMI_CLK_CS        0x00000E04
#define CHAR_LENGTH             0x8

/******************************************************************************
**                      INTERNAL FUNCTION PROTOTYPES
*******************************************************************************/
static void SPIConfigDataFmtReg(unsigned int dataFormat);
static void SPIInitialize(void);
static void SetUpInt(void);
static void SPISetUp(void);
static void GetStatusCommand(volatile char *);
static void ResetCommand(void);
static void SpiTxEdmaParamSet(unsigned int tccNum, 
                              unsigned int chNum,
                              volatile char *buffer, 
                              unsigned int buffLength);
static void SpiRxEdmaParamSet(unsigned int tccNum, 
                              unsigned int chNum,
                              volatile char *buffer, 
                              unsigned int length, 
                              unsigned int destBIdxFlag);

static void callback(unsigned int tccNum, 
                     unsigned int status);
static void Edma3CCErrHandlerIsr(void); 
static void Edma3ComplHandlerIsr(void);
static void RequestEDMA3Channels(void);
static void EDMA3Initialize(void);
static void ConfigureIntEDMA3(void);
static void CSHoldAssert(void);
static void CSHoldDeassert(void);

/******************************************************************************
**                      INTERNAL VARIABLE DEFINITIONS
*******************************************************************************/
static void (*cb_Fxn[EDMA3_NUM_TCC]) (unsigned int tcc,
                                      unsigned int status);
volatile unsigned int flagTx = 0;
volatile unsigned int flagRx = 0;
unsigned int evtQ = 0;

/******************************************************************************
**                      INTERNAL FUNCTION DEFINITIONS
*******************************************************************************/
int main(void)
{
	volatile char statusResponse[23] = {0};

	flagTx = 0;
	flagRx = 0;

    /* Initializing the UART instance for serial communication. */
    UARTStdioInit();

    UARTPuts("Welcome to StarterWare SPI application.\r\n\r\n", -1);
    UARTPuts("Here the SPI controller on the SoC communicates with", -1);
    UARTPuts(" the Fingerprint Sensor present on the LCDK.\r\n\r\n", -1);

    /* Initializes the EDMA3 Instance. */
    EDMA3Initialize();

    /* Initializes the SPI1 Instance. */
    SPIInitialize();

    /* Request EDMA3CC for Tx and Rx channels for SPI1. */
    RequestEDMA3Channels();

    /* Enable SPI communication */
    SPIEnable(SOC_SPI_1_REGS);

    /* Issue a Reset to the Fingerprint Sensor */
    ResetCommand();
    UARTPuts("Attempting to Reset the Fingerprint Sensor.\r\n\r\n", -1);

    /* Read the Status of the Fingerprint Sensor */
    GetStatusCommand(statusResponse);
    if(statusResponse[18] == 0x48)
    {
        UARTPuts("Fingerprint Sensor Successfully Reset.\r\n\r\n", -1);
    }
    else
    {
        UARTPuts("ERROR: Fingerprint Sensor Reset Attempt Unsuccessful.\r\n\r\n", -1);
    }

    while(1);
}


/*
** This function is used to set the PaRAM entries of EDMA3 for the Transmit
** Channel of SPI1. The corresponding EDMA3 channel is also enabled for
** transmission.
*/
static void SpiTxEdmaParamSet(unsigned int tccNum, unsigned int chNum,
                              volatile char *buffer, unsigned int buffLength)
{
    EDMA3CCPaRAMEntry paramSet;
    unsigned char *p = (unsigned char *)&paramSet;
    unsigned int i = 0;

    /* Clean-up the contents of structure variable. */
    for (i = 0; i < sizeof(paramSet); i++)
    {
        p[i] = 0;
    }
    /* Fill the PaRAM Set with transfer specific information. */

    /* srcAddr holds address of memory location buffer. */
    paramSet.srcAddr = (unsigned int) buffer;

    /* destAddr holds address of SPIDAT1 register. */
    paramSet.destAddr = (unsigned int) (SOC_SPI_1_REGS + SPI_SPIDAT1);

    /* aCnt holds the number of bytes in an array. */
    paramSet.aCnt = (unsigned short) 1;

    /* bCnt holds the number of such arrays to be transferred. */
    paramSet.bCnt = (unsigned short) buffLength;

    /* cCnt holds the number of frames of aCnt*bBcnt bytes to be transferred. */
    paramSet.cCnt = (unsigned short) 1;

    /*
    ** The srcBidx should be incremented by aCnt number of bytes since the
    ** source used here is  memory.
    */
    paramSet.srcBIdx = (short) 1;

    /* A sync Transfer Mode is set in OPT.*/
    /* srCIdx and destCIdx set to zero since ASYNC Mode is used. */
    paramSet.srcCIdx = (short) 0;

    /* Linking transfers in EDMA3 are not used. */
    paramSet.linkAddr = (unsigned short)0xFFFF;
    paramSet.bCntReload = (unsigned short)0;

    paramSet.opt = 0x00000000;

    /* SAM field in OPT is set to zero since source is memory and memory
       pointer needs to be incremented. DAM field in OPT is set to zero
       since destination is not a FIFO. */

    /* Set TCC field in OPT with the tccNum. */
    paramSet.opt |= ((tccNum << EDMA3CC_OPT_TCC_SHIFT) & EDMA3CC_OPT_TCC);

    /* EDMA3 Interrupt is enabled and Intermediate Interrupt Disabled.*/
    paramSet.opt |= (1 << EDMA3CC_OPT_TCINTEN_SHIFT);

    /* Now write the PaRam Set to EDMA3.*/
    EDMA3SetPaRAM(SOC_EDMA30CC_0_REGS, chNum, &paramSet);

    /* EDMA3 Transfer is Enabled. */
    EDMA3EnableTransfer(SOC_EDMA30CC_0_REGS, chNum, EDMA3_TRIG_MODE_EVENT);
}

/*
** This function is used to set the PaRAM entries of EDMA3 for the Receive
** Channel of SPI1. The corresponding EDMA3 channel is also enabled for
** reception.
*/

static void SpiRxEdmaParamSet(unsigned int tccNum, unsigned int chNum,
                              volatile char *buffer, unsigned int length,
                              unsigned int destBidxFlag)
{
    EDMA3CCPaRAMEntry paramSet;
    unsigned char *p = (unsigned char *)&paramSet;
    unsigned int i = 0;

    /* Clean-up the contents of structure variable. */
    for (i = 0; i < sizeof(paramSet); i++)
    {
        p[i] = 0;
    }

    /* Fill the PaRAM Set with Receive specific information. */

    /* srcAddr holds address of SPI Rx FIFO. */
    paramSet.srcAddr = (unsigned int) (SOC_SPI_1_REGS + SPI_SPIBUF);

    /* destAddr is address of memory location named buffer. */
    paramSet.destAddr = (unsigned int) buffer;

    /* aCnt holds the number of bytes in an array. */
    paramSet.aCnt = (unsigned short) 1;

    /* bCnt holds the number of such arrays to be transferred. */
    paramSet.bCnt = (unsigned short) length;

    /* cCnt holds the number of frames of aCnt*bBcnt bytes to be transferred. */
    paramSet.cCnt = (unsigned short) 1;

    /* The srcBidx should not be incremented since it is a h/w register. */
    paramSet.srcBIdx = 0;

    if(TRUE == destBidxFlag)
    {
        /* The destBidx should be incremented for every byte. */
        paramSet.destBIdx = 1;
    }
    else
    {
        /* The destBidx should not be incremented. */
        paramSet.destBIdx = 0;
    }

    /* A sync Transfer Mode. */
    /* srCIdx and destCIdx set to zero since ASYNC Mode is used. */
    paramSet.srcCIdx = 0;
    paramSet.destCIdx = 0;

    /* Linking transfers in EDMA3 are not used. */
    paramSet.linkAddr = (unsigned short)0xFFFF;
    paramSet.bCntReload = 0;

    paramSet.opt = 0x00000000;

    /* Set TCC field in OPT with the tccNum. */
    paramSet.opt |= ((tccNum << EDMA3CC_OPT_TCC_SHIFT) & EDMA3CC_OPT_TCC);

    /* EDMA3 Interrupt is enabled and Intermediate Interrupt Disabled.*/
    paramSet.opt |= (1 << EDMA3CC_OPT_TCINTEN_SHIFT);

    /* Now write the PaRam Set to EDMA3.*/
    EDMA3SetPaRAM(SOC_EDMA30CC_0_REGS, chNum, &paramSet);

    /* EDMA3 Transfer is Enabled. */
    EDMA3EnableTransfer(SOC_EDMA30CC_0_REGS, chNum, EDMA3_TRIG_MODE_EVENT);
}

/*
** EDMA3 completion Interrupt Service Routine(ISR).
*/

static void Edma3ComplHandlerIsr(void)
{
    volatile unsigned int pendingIrqs;
    volatile unsigned int isIPR = 0;

    volatile unsigned int indexl;
    volatile unsigned int Cnt = 0;
    indexl = 1;
#ifdef _TMS320C6X
    IntEventClear(SYS_INT_EDMA3_0_CC0_INT1);
#else
    IntSystemStatusClear(SYS_INT_CCINT0);
#endif
    isIPR = EDMA3GetIntrStatus(SOC_EDMA30CC_0_REGS);
    if(isIPR)
    {
        while ((Cnt < EDMA3CC_COMPL_HANDLER_RETRY_COUNT)&& (indexl != 0))
        {
            indexl = 0;
            pendingIrqs = EDMA3GetIntrStatus(SOC_EDMA30CC_0_REGS);
            while (pendingIrqs)
            {
                if((pendingIrqs & 1) == TRUE)
                {
                    /**
                     * If the user has not given any callback function
                     * while requesting the TCC, its TCC specific bit
                     * in the IPR register will NOT be cleared.
                     */
                    /* Here write to ICR to clear the corresponding IPR bits. */
                    EDMA3ClrIntr(SOC_EDMA30CC_0_REGS, indexl);
                    (*cb_Fxn[indexl])(indexl, EDMA3_XFER_COMPLETE);
                }
                ++indexl;
                pendingIrqs >>= 1;
            }
            Cnt++;
        }
    }

}


/*
** EDMA3 Error Interrupt Service Routine(ISR).
*/

static void Edma3CCErrHandlerIsr(void)
{
    volatile unsigned int pendingIrqs = 0;
    unsigned int Cnt = 0;
    unsigned int index = 1;
    unsigned int regionNum = 0;
    unsigned int evtqueNum = 0;

#ifdef _TMS320C6X
    IntEventClear(SYS_INT_EDMA3_0_CC0_ERRINT);
#else
    IntSystemStatusClear(SYS_INT_CCERRINT);
#endif

    if((HWREG(SOC_EDMA30CC_0_REGS + EDMA3CC_EMR) != 0 )
        || (HWREG(SOC_EDMA30CC_0_REGS + EDMA3CC_QEMR) != 0)
        || (HWREG(SOC_EDMA30CC_0_REGS + EDMA3CC_CCERR) != 0))
    {
        /* Loop for EDMA3CC_ERR_HANDLER_RETRY_COUNT number of time, breaks
           when no pending interrupt is found. */
        while ((Cnt < EDMA3CC_ERR_HANDLER_RETRY_COUNT) && (index != 0))
        {
            index = 0;
            pendingIrqs = HWREG(SOC_EDMA30CC_0_REGS + EDMA3CC_EMR);
            while (pendingIrqs)
            {
                /*Process all the pending interrupts.*/
                if((pendingIrqs & 1)==TRUE)
                {
                    /* Write to EMCR to clear the corresponding EMR bits.*/
                    HWREG(SOC_EDMA30CC_0_REGS + EDMA3CC_EMCR) = (1<<index);
                    /*Clear any SER*/
                    HWREG(SOC_EDMA30CC_0_REGS + EDMA3CC_S_SECR(regionNum)) = (1<<index);
                }
                ++index;
                pendingIrqs >>= 1;
            }
            index = 0;
            pendingIrqs = HWREG(SOC_EDMA30CC_0_REGS + EDMA3CC_QEMR);
            while (pendingIrqs)
            {
                /*Process all the pending interrupts.*/
                if((pendingIrqs & 1)==TRUE)
                {
                    /* Here write to QEMCR to clear the corresponding QEMR bits.*/
                    HWREG(SOC_EDMA30CC_0_REGS + EDMA3CC_QEMCR) = (1<<index);
                    /*Clear any QSER*/
                    HWREG(SOC_EDMA30CC_0_REGS + EDMA3CC_S_QSECR(0)) = (1<<index);
                }
                ++index;
                pendingIrqs >>= 1;
            }
            index = 0;
            pendingIrqs = HWREG(SOC_EDMA30CC_0_REGS + EDMA3CC_CCERR);
            if(pendingIrqs != 0)
            {
                /* Process all the pending CC error interrupts. */
                /* Queue threshold error for different event queues.*/
                for (evtqueNum = 0; evtqueNum < EDMA3_0_NUM_EVTQUE; evtqueNum++)
                {
                    if((pendingIrqs & (1 << evtqueNum)) != 0)
                    {
                        /* Clear the error interrupt. */
                        HWREG(SOC_EDMA30CC_0_REGS + EDMA3CC_CCERRCLR) = (1 << evtqueNum);
                    }
                }

                /* Transfer completion code error. */
                if ((pendingIrqs & (1 << EDMA3CC_CCERR_TCCERR_SHIFT)) != 0)
                {
                    HWREG(SOC_EDMA30CC_0_REGS + EDMA3CC_CCERRCLR) = \
                         (0x01 << EDMA3CC_CCERR_TCCERR_SHIFT);
                }
                ++index;
            }
            Cnt++;
        }
    }
}

/*
** This function is used as a callback from EDMA3 Completion Handler.
** The DMA Mode operation of SPI1 is disabled over here.
*/
static void callback(unsigned int tccNum, unsigned int status)
{
    if(tccNum == 19)
    {
        flagTx = 1;
        /* Disable SPI-EDMA Communication. */
        SPIIntDisable(SOC_SPI_1_REGS, SPI_DMA_REQUEST_ENA_INT);
    }
    else if(tccNum == 18)
    {
        flagRx = 1;
        /* Disable SPI-EDMA Communication. */
        SPIIntDisable(SOC_SPI_1_REGS, SPI_DMA_REQUEST_ENA_INT);
    }
}

/*
** This function configures the AINTC to receive EDMA3 interrupts.
*/
static void ConfigureIntEDMA3(void)
{
#ifdef _TMS320C6X
	IntRegister(C674X_MASK_INT4, Edma3ComplHandlerIsr);
	IntRegister(C674X_MASK_INT5, Edma3CCErrHandlerIsr);

	IntEventMap(C674X_MASK_INT4, SYS_INT_EDMA3_0_CC0_INT1);
	IntEventMap(C674X_MASK_INT5, SYS_INT_EDMA3_0_CC0_ERRINT);

	IntEnable(C674X_MASK_INT4);
	IntEnable(C674X_MASK_INT5);
#else
    IntRegister(SYS_INT_CCINT0, Edma3ComplHandlerIsr);

    IntChannelSet(SYS_INT_CCINT0, 2);

    IntSystemEnable(SYS_INT_CCINT0);

    IntRegister(SYS_INT_CCERRINT, Edma3CCErrHandlerIsr);

    IntChannelSet(SYS_INT_CCERRINT, 2);

    IntSystemEnable(SYS_INT_CCERRINT);
#endif
}


/*
** This function allocates EDMA3 channels to SPI1 for trasmisssion and
** reception purposes.
*/
static void RequestEDMA3Channels(void)
{
    /* Request DMA Channel and TCC for SPI Transmit*/
    EDMA3RequestChannel(SOC_EDMA30CC_0_REGS, EDMA3_CHANNEL_TYPE_DMA, \
                        EDMA3_CHA_SPI1_TX, EDMA3_CHA_SPI1_TX, evtQ);

    /* Request DMA Channel and TCC for SPI Receive*/
    EDMA3RequestChannel(SOC_EDMA30CC_0_REGS, EDMA3_CHANNEL_TYPE_DMA, \
                        EDMA3_CHA_SPI1_RX, EDMA3_CHA_SPI1_RX, evtQ);
}

/*
** This function asserts the CSHOLD line.
*/

static void CSHoldAssert(void)
{
    SPIDat1Config(SOC_SPI_1_REGS, (SPI_CSHOLD | SPI_DATA_FORMAT0), 0x04);
}


/*
** This function deasserts the CSHOLD line.
*/

static void CSHoldDeassert(void)
{
    SPIDat1Config(SOC_SPI_1_REGS, SPI_DATA_FORMAT0, 0x04);
}

static void GetStatusCommand(volatile char *pStatusResponse)
{
    volatile char getStatusCommand[23] = {0};
    unsigned int buffLength = 23;

	// Issue Get Status command and read Status Message Response
    getStatusCommand[0] = 0x80;
    getStatusCommand[1] = 0x10;
    getStatusCommand[2] = 0xBA;
    getStatusCommand[3] = 0x07;
    getStatusCommand[4] = 0x00;
    getStatusCommand[5] = 0x00;
    getStatusCommand[6] = 0x00;
    getStatusCommand[7] = 0xFF;
    getStatusCommand[8] = 0xFF;
    getStatusCommand[9] = 0xFF;
    getStatusCommand[10] = 0xFF;
    getStatusCommand[11] = 0xFF;
    getStatusCommand[12] = 0xFF;
    getStatusCommand[13] = 0xFF;
    getStatusCommand[14] = 0xFF;
    getStatusCommand[15] = 0xFF;
    getStatusCommand[16] = 0xFF;
    getStatusCommand[17] = 0xFF;
    getStatusCommand[18] = 0xFF;
    getStatusCommand[19] = 0xFF;
    getStatusCommand[20] = 0xFF;
    getStatusCommand[21] = 0xFF;
    getStatusCommand[22] = 0xFF;

    /* Configure the PaRAM registers in EDMA for Transmission.*/
    SpiTxEdmaParamSet(EDMA3_CHA_SPI1_TX, EDMA3_CHA_SPI1_TX, getStatusCommand, buffLength);

    /* Registering Callback Function for Transmission. */
    cb_Fxn[EDMA3_CHA_SPI1_TX] = &callback;

    /* Configure the PaRAM registers in EDMA for Reception.*/
    SpiRxEdmaParamSet(EDMA3_CHA_SPI1_RX, EDMA3_CHA_SPI1_RX, pStatusResponse, buffLength, TRUE);

    /* Registering Callback Function for Reception. */
    cb_Fxn[EDMA3_CHA_SPI1_RX] = &callback;

    /* Assert the CSHOLD line corresponding to the SPI Flash. */
    CSHoldAssert();

    /* Enable SPI controller to generate DMA events */
    SPIIntEnable(SOC_SPI_1_REGS, SPI_DMA_REQUEST_ENA_INT);

    /* Wait until both the flags are set to 1 in the callback function. */
    while((0 == flagTx) || (0 == flagRx));

    flagTx = 0;
    flagRx = 0;

    /* Deassert the CSHOLD line corresponding to the SPI Flash. */
    CSHoldDeassert();
}

static void ResetCommand(void)
{
    volatile char resetCommand[7] = {0};
    unsigned int buffLength = 7;
    volatile char dummy;

    resetCommand[0] = 0x80;
    resetCommand[1] = 0x14;
    resetCommand[2] = 0xBA;
    resetCommand[3] = 0x07;
    resetCommand[4] = 0x00;
    resetCommand[5] = 0x00;
    resetCommand[6] = 0x00;

    /* Configure the PaRAM registers in EDMA for Transmission. */
    SpiTxEdmaParamSet(EDMA3_CHA_SPI1_TX, EDMA3_CHA_SPI1_TX, resetCommand, buffLength);

    /* Registering Callback Function for Transmission. */
    cb_Fxn[EDMA3_CHA_SPI1_TX] = &callback;

    /* Configure the PaRAM registers in EDMA for Reception. */
    SpiRxEdmaParamSet(EDMA3_CHA_SPI1_RX, EDMA3_CHA_SPI1_RX, &dummy, buffLength, FALSE);

    /* Registering Callback Function for Reception. */
    cb_Fxn[EDMA3_CHA_SPI1_RX] = &callback;

    /* Assert the CSHOLD line corresponding to the SPI Flash. */
    CSHoldAssert();

    /* Enable SPI controller to generate DMA events */
    SPIIntEnable(SOC_SPI_1_REGS, SPI_DMA_REQUEST_ENA_INT);

    /* Wait until both the flags are set to 1 in the callback function. */
    while((0 == flagTx) || (0 == flagRx));

    flagTx = 0;
    flagRx = 0;

    /* Deassert the CSHOLD line corresponding to the SPI Flash. */
    CSHoldDeassert();
}

/*
** Configures ARM interrupt controller to generate SPI interrupt
**
*/
static void SetUpInt(void)
{
	// Setup the ARM or DSP interrupt controller

#ifdef _TMS320C6X
	// Initialize DSP interrupt controller
	IntDSPINTCInit();

	// Enable DSP interrupts globally
	IntGlobalEnable();
#else
    /*Initialize the ARM Interrupt Controller(AINTC). */
    IntAINTCInit();

    /* Enable IRQ in CPSR.*/
    IntMasterIRQEnable();

    /* Enable the interrupts in GER of AINTC.*/
    IntGlobalEnable();

    /* Enable the interrupts in HIER of AINTC.*/
    IntIRQEnable();
#endif
}

/*
** Configures Data Format register of SPI
**
*/
static void SPIConfigDataFmtReg(unsigned int dataFormat)
{
    SPIConfigClkFormat(SOC_SPI_1_REGS,
                       (SPI_CLK_POL_LOW | SPI_CLK_INPHASE),
                       dataFormat);

    /* Configures SPI to transmit MSB bit First during data transfer */
    SPIShiftMsbFirst(SOC_SPI_1_REGS, dataFormat);

    /* Sets the Charcter length */
    SPICharLengthSet(SOC_SPI_1_REGS, CHAR_LENGTH, dataFormat);
}

/*
** This function configures the power supply for EDMA3 Channel Controller 0
** and Transfer Controller 0, registers the EDMA interrupts in AINTC.
*/
static void EDMA3Initialize(void)
{
    /* Enabling the PSC for EDMA3CC_0).*/
    PSCModuleControl(SOC_PSC_0_REGS, HW_PSC_CC0, PSC_POWERDOMAIN_ALWAYS_ON,
		     PSC_MDCTL_NEXT_ENABLE);

    /* Enabling the PSC for EDMA3TC_0.*/
    PSCModuleControl(SOC_PSC_0_REGS, HW_PSC_TC0, PSC_POWERDOMAIN_ALWAYS_ON,
		     PSC_MDCTL_NEXT_ENABLE);

    /* Initialization of EDMA3 */
    EDMA3Init(SOC_EDMA30CC_0_REGS, evtQ);

    /*
    ** Enable AINTC to handle interuppts. Also enable IRQ interuppt in
    ** ARM processor.
    */
    SetUpInt();

    /* Register EDMA3 Interrupts */
    ConfigureIntEDMA3();
}

/*
** Initialization of SPI1 Instance is done here.
*/
static void SPIInitialize(void)
{
    PSCModuleControl(SOC_PSC_1_REGS, HW_PSC_SPI1, PSC_POWERDOMAIN_ALWAYS_ON,
		     PSC_MDCTL_NEXT_ENABLE);

    /* Using SPI1 instance. */
    SPIPinMuxSetup(1);

    /* Select CS0 of SPI1. The SPI Flash is connected to SPI1_SCS[0]. */
    SPI1CSPinMuxSetup(2);

    SPISetUp();
}

/*
** Configures the SPI1 instance for communication.
*/
static void SPISetUp(void)
{
    unsigned char dcs = 0x04;
    unsigned int val = SIMO_SOMI_CLK_CS;

    /* Resets the SPI */
    SPIReset(SOC_SPI_1_REGS);

    /* Brings SPI Out-of-Reset */
    SPIOutOfReset(SOC_SPI_1_REGS);

    /* Configures SPI in Master Mode */
    SPIModeConfigure(SOC_SPI_1_REGS, SPI_MASTER_MODE);

    /* Sets SPI Controller for 4-pin Mode with Chip Select */
    SPIPinControl(SOC_SPI_1_REGS, 0, 0, &val);

    /* Configures the Prescale bit in Data Format register. */
    SPIClkConfigure(SOC_SPI_1_REGS, 150000000, 1000000,
                   SPI_DATA_FORMAT0);

    /* Chip Select Default Pattern is Set To 1 in SPIDEF Register*/
    SPIDefaultCSSet(SOC_SPI_1_REGS, dcs);

    /* Configures SPI Data Format Register */
    SPIConfigDataFmtReg(SPI_DATA_FORMAT0);
}
/******************************* End of file *********************************/
