--- Title:IndustrialIO_top.vhd
--- Description: 
---
---     o  0
---     | /       Copyright (c) 2012
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 10/30/2012
--- Version: 1.00
---
--- Revision History
---   1.00 - Initial revision
---   1.01 - add support for DAWG
---   1.02 - add support for HS ADC
---   1.03 - fix up pin assignments for 2 UPP ports.
---   1.04 - use ADC clock instead of EMIFA clock

library WORK;
library IEEE;
library UNISIM;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use UNISIM.VCOMPONENTS.ALL;
use WORK.MityDSP_L138_pkg.ALL;

entity IndustrialIO_top is
   generic ( DECODE_BITS   : integer range 1 to 9 := 5
   );
   port (
      i_ema_clk	      : in std_logic; -- 100 MHz EMIF clock

      -- DSP EMIFA BUS Interface
      i_ema_cs0_n   : in std_logic; -- RESERVED for SDRAM Controller
      i_ema_cs2_n   : in std_logic; -- ARM core CE space
      i_ema_cs3_n   : in std_logic; -- NAND FLASH space, not used
      i_ema_cs4_n   : in std_logic; -- DSP core CE space
      i_ema_cs5_n   : in std_logic; -- Reserved core space, not used
      i_ema_oe_n    : in std_logic;
      i_ema_we_n    : in std_logic;
      io_ema_wait_n : inout std_logic_vector(1 downto 0);
      io_ema_d      : inout std_logic_vector(15 downto 0);
      i_ema_a       : in std_logic_vector(13 downto 0);
      i_ema_ba      : in std_logic_vector(1 downto 0);
      i_ema_wen_dqm : in std_logic_vector(1 downto 0); 
      i_ema_cas     : in std_logic; -- reserved for SDRAM controller, not used
      i_ema_ras     : in std_logic; -- reserved for SDRAM controller, not used
      i_ema_sdcke   : in std_logic; -- reserved for SDRAM controller, not used
      i_ema_rnw     : in std_logic; -- reserved for SDRAM controller, not used
                        
      -- Analog Expansion - ADS1158
      o_ads1158_clkio  : out std_logic;  -- CLKIO
      o_ads1158_sclk   : out std_logic;  -- SCLK
      o_ads1158_din    : out std_logic;  -- DIN
      i_ads1158_dout   : in  std_logic;  -- DOUT
      o_ads1158_cs_n   : out std_logic;  -- CS
      o_ads1158_start  : out std_logic;  -- START
      i_ads1158_drdy   : in  std_logic;  -- DRDY

      -- Analog Expansion - DAC5672
      o_clk1          : out std_logic;
      o_da            : out std_logic_vector(13 downto 0);   
      o_clk2          : out std_logic;
      o_db            : out std_logic_vector(13 downto 0);
      
      -- Analog Expansion - ADS5560
      i_adc_in_p      : in  std_logic_vector(7 downto 0);  -- DDR inputs
      i_adc_in_n      : in  std_logic_vector(7 downto 0);  -- DDR inputs
      i_adc_clk_p     : in  std_logic;
      i_adc_clk_n     : in  std_logic;
      i_adc_ovr       : in  std_logic;

      -- UPP port DAWG data (channel A, [XDATA[7:0] Data[7:0]]) 
      i_upp_cha_clock : in  std_logic;
      i_upp_xd        : in  std_logic_vector(7 downto 0);
      i_upp_d         : in  std_logic_vector(7 downto 0);
      i_upp_cha_enable: in  std_logic;
      i_upp_cha_start : in  std_logic;
      o_upp_cha_wait  : out std_logic;
      
      -- UPP port ADC data (channel B, [XData[15:8] Data[15:8]])
      o_upp_chb_clock  : out std_logic;
      o_upp_xd         : out std_logic_vector(15 downto 8);
      o_upp_d          : out std_logic_vector(15 downto 8);
      o_upp_chb_enable : out std_logic;
      o_upp_chb_start  : out std_logic;
      i_upp_chb_wait   : in  std_logic; 
      
      -- DSP IRQ lines
      o_nmi_n       : out std_logic;
      o_int         : out std_logic_vector(1 downto 0)
                  
  );
end IndustrialIO_top;

--------------------------------------------------------------------------
-- ARCHITECTURE
--------------------------------------------------------------------------
architecture rtl of IndustrialIO_top is

constant FPGA_APPLICATION_ID: std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR( 00, 8);
constant FPGA_VERSION_MAJOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 01, 4);
constant FPGA_VERSION_MINOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 04, 4);
constant FPGA_YEAR:           std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 14, 5);
constant FPGA_MONTH:          std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 06, 4);
constant FPGA_DAY:            std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 30, 5);

constant CORE_BASE_MODULE: integer := 0;
constant CORE_GPIO_MODULE   : integer := 1;
constant CORE_SPI_MODULE    : integer := 2;
constant CORE_DAWG_MODULE   : integer := 3;
constant CORE_HSADC_MODULE  : integer := 4;

constant CORE_GPIO_IRQ_LEVEL : integer := 0;
constant CORE_GPIO_IRQ_VECTOR : integer := 1;
constant CORE_SPI_IRQ_LEVEL   : integer := 0;
constant CORE_SPI_IRQ_VECTOR  : integer := 2;
constant CORE_DAWG_IRQ_LEVEL   : integer := 1;
constant CORE_DAWG_IRQ_VECTOR  : integer := 3;
constant CORE_HSADC_IRQ_LEVEL   : integer := 1;
constant CORE_HSADC_IRQ_VECTOR  : integer := 4;

component fifo_dpram64x32
	port (
	clk: IN std_logic;
	rst: IN std_logic;
	din: IN std_logic_VECTOR(31 downto 0);
	wr_en: IN std_logic;
	rd_en: IN std_logic;
	dout: OUT std_logic_VECTOR(31 downto 0);
	full: OUT std_logic;
	empty: OUT std_logic;
	data_count: OUT std_logic_VECTOR(5 downto 0));
end component; 

component dual_awg_dac5672
   Port ( 
      clk             : in  std_logic;
      i_ABus          : in  std_logic_vector(5 downto 0);
      i_DBus          : in  std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in  std_logic;
      i_rd_en         : in  std_logic;
      i_cs            : in  std_logic;
      o_irq           : out std_logic := '0';
      i_ilevel        : in  std_logic_vector(1 downto 0) := "00";      
      i_ivector       : in  std_logic_vector(3 downto 0) := "0000";   
      
      i_cha_clock     : in  std_logic;
      i_cha_data      : in  std_logic_vector(15 downto 0);
      i_cha_enable    : in  std_logic;
      i_cha_start     : in  std_logic;
      o_cha_wait      : out std_logic;
      
      -- DAV5672 interface
      i_dac_clk       : in std_logic;
      o_clka          : out std_logic;
      o_da            : out std_logic_vector(13 downto 0);   
      o_clkb          : out std_logic;
      o_db            : out std_logic_vector(13 downto 0)
      
   );
end component;

component ads5560
   Port ( 
      clk             : in  std_logic;
      i_ABus          : in  std_logic_vector(5 downto 0);
      i_DBus          : in  std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in  std_logic;
      i_rd_en         : in  std_logic;
      i_cs            : in  std_logic;
      o_irq           : out std_logic := '0';
      i_ilevel        : in  std_logic_vector(1 downto 0) := "00";      
      i_ivector       : in  std_logic_vector(3 downto 0) := "0000";   
      
      -- ADC interface
      i_adc_in_p      : in  std_logic_vector(7 downto 0);  -- DDR inputs
      i_adc_in_n      : in  std_logic_vector(7 downto 0);  -- DDR inputs
      i_adc_clk_p     : in  std_logic;
      i_adc_clk_n     : in  std_logic;
      i_adc_ovr       : in  std_logic;
		o_adc_clk       : out std_logic;
      
      -- UPP interface
      i_upp_clock      : in  std_logic; -- from fabric, minimum 40 MHz
      o_upp_clock      : out std_logic;
      o_upp_d          : out std_logic_vector(15 downto 0);
      o_upp_enable     : out std_logic;
      o_upp_start      : out std_logic;
      i_upp_wait       : in  std_logic
                  
   );
end component;

-- clock/reset related signals
signal clk25mhz: std_logic;
signal ema_clk: std_logic;
signal dcm_lock: std_logic;
signal dcm_reset: std_logic;
signal dcm_status: std_logic_vector(2 downto 0);
signal wd_rst: std_logic;

-- EMIFA interface signals
signal ema_d : std_logic_vector(15 downto 0);
signal ema_wait_n : std_logic_vector(1 downto 0) := "11";
signal t_ema_wait : std_logic;
signal t_ema_d : std_logic;
signal be_r : std_logic_vector(1 downto 0);
signal addr_r : std_logic_vector(5 downto 0);
signal arm_cs5_r : std_logic_vector((2**DECODE_BITS)-1 downto 0);
signal dsp_cs4_r : std_logic_vector((2**DECODE_BITS)-1 downto 0);
signal edi_r : std_logic_vector(15 downto 0);
signal edo_arm : bus16_vector((2**DECODE_BITS)-1 downto 0);
signal edo_dsp : bus16_vector((2**DECODE_BITS)-1 downto 0);
signal rd_r : std_logic;
signal wr_r : std_logic;
signal irq_map  : bus16_vector(1 downto 0) := (others=>(others=>'0')); 

-- ADS1158 SPI
signal mosi_din : std_logic_VECTOR(31 downto 0);
signal mosi_wr_en : std_logic;
signal mosi_rd_en : std_logic;
signal mosi_dout  : std_logic_VECTOR(31 downto 0);
signal mosi_full : std_logic;
signal mosi_empty : std_logic;
signal mosi_wr_data_count : std_logic_VECTOR(11 downto 0) := (others=>'0');
signal spi_fifo_rst : std_logic := '0';

signal miso_wr_clk : std_logic;
signal miso_rd_clk : std_logic;
signal miso_din : std_logic_VECTOR(31 downto 0);
signal miso_wr_en : std_logic;
signal miso_rd_en : std_logic;
signal miso_dout  : std_logic_VECTOR(31 downto 0);
signal miso_empty : std_logic;
signal miso_rd_data_count : std_logic_VECTOR(11 downto 0) := (others=>'0');
signal spi_cs : std_logic_vector(7 downto 0) := (others=>'1');

signal gpio_in, gpio_t, gpio_out : std_logic_vector(15 downto 0) := (others=>'0');

signal ads1158_cnt : std_logic_vector(4 downto 0) := (others => '0');
signal ads1158_clk : std_logic := '0';
signal clk1, clk2 : std_logic := '0';

signal upp_clock, upp_clock_out : std_logic := '0';
signal adc_clk : std_logic := '0';

begin -- architecture: rtl

--------------------------------------------------------------------------
-- System DCM Instantiation
--------------------------------------------------------------------------
--------------------------------------------------------------------------
-- Clock Buffer Instantiation
--------------------------------------------------------------------------
-- emaclkbuf : bufg port map (O => ema_clk, I => i_ema_clk);
emaclkdcm : DCM_SP
   generic map (
      CLKDV_DIVIDE => 2.0,                   -- CLKDV divide value
                                             -- (1.5,2,2.5,3,3.5,4,4.5,5,5.5,6,6.5,7,7.5,8,9,10,11,12,13,14,15,16).
      CLKFX_DIVIDE => 10,                     -- Divide value on CLKFX outputs - D - (1-32)
      CLKFX_MULTIPLY => 4,                   -- Multiply value on CLKFX outputs - M - (2-32)
      CLKIN_DIVIDE_BY_2 => FALSE,            -- CLKIN divide by two (TRUE/FALSE)
      CLKIN_PERIOD => 10.0,                  -- Input clock period specified in nS
      CLKOUT_PHASE_SHIFT => "NONE",          -- Output phase shift (NONE, FIXED, VARIABLE)
      CLK_FEEDBACK => "1X",                  -- Feedback source (NONE, 1X, 2X)
      DESKEW_ADJUST => "SYSTEM_SYNCHRONOUS", -- SYSTEM_SYNCHRNOUS or SOURCE_SYNCHRONOUS
      DFS_FREQUENCY_MODE => "LOW",           -- Unsupported - Do not change value
      DLL_FREQUENCY_MODE => "LOW",           -- Unsupported - Do not change value
      DSS_MODE => "NONE",                    -- Unsupported - Do not change value
      DUTY_CYCLE_CORRECTION => TRUE,         -- Unsupported - Do not change value
      FACTORY_JF => X"c080",                 -- Unsupported - Do not change value
      PHASE_SHIFT => 0,                      -- Amount of fixed phase shift (-255 to 255)
      STARTUP_WAIT => FALSE                  -- Delay config DONE until DCM_SP LOCKED (TRUE/FALSE)
   )
   port map (
      CLK0     => ema_clk,         -- 1-bit output: 0 degree clock output
      CLK180   => open,            -- 1-bit output: 180 degree clock output
      CLK270   => open,            -- 1-bit output: 270 degree clock output
      CLK2X    => open,            -- 1-bit output: 2X clock frequency clock output
      CLK2X180 => open,            -- 1-bit output: 2X clock frequency, 180 degree clock output
      CLK90    => open,            -- 1-bit output: 90 degree clock output
      CLKDV    => upp_clock,       -- 1-bit output: Divided clock output
      CLKFX    => adc_clk,         -- 1-bit output: Digital Frequency Synthesizer output (DFS)
      CLKFX180 => open,            -- 1-bit output: 180 degree CLKFX output
      LOCKED   => dcm_lock,        -- 1-bit output: DCM_SP Lock Output
      PSDONE   => open,            -- 1-bit output: Phase shift done output
      STATUS(2 downto 0) => dcm_status,     -- 8-bit output: DCM_SP status output
      STATUS(7 downto 3) => open,
      CLKFB    => ema_clk,       -- 1-bit input: Clock feedback input
      CLKIN    => i_ema_clk,     -- 1-bit input: Clock input
      DSSEN    => '0',           -- 1-bit input: Unsupported, specify to GND.
      PSCLK    => '0',           -- 1-bit input: Phase shift clock input
      PSEN     => '0',           -- 1-bit input: Phase shift enable
      PSINCDEC => '0',           -- 1-bit input: Phase shift increment/decrement input
      RST      => dcm_reset      -- 1-bit input: Active high reset input
   );
------------------------------------------------------------------------------
-- EMIF Interface Module: 
------------------------------------------------------------------------------
emififace : EMIFA_iface
    generic map ( 
        DECODE_BITS       => DECODE_BITS, 
        CONFIG            => "MityDSP_L138" 
    )
    port map ( ema_clk   => ema_clk, 
           -- EMIFA signals
           i_ema_cs0_n   => i_ema_cs0_n, 
           i_ema_cs2_n   => i_ema_cs2_n, 
           i_ema_cs3_n   => i_ema_cs3_n, 
           i_ema_cs4_n   => i_ema_cs4_n, 
           i_ema_cs5_n   => i_ema_cs5_n, 
           i_ema_oe_n    => i_ema_oe_n, 
           i_ema_we_n    => i_ema_we_n, 
           o_ema_wait_n  => ema_wait_n, 
           t_ema_wait    => t_ema_wait, 
           i_ema_d       => io_ema_d, 
           o_ema_d       => ema_d, 
           t_ema_d       => t_ema_d, 
           i_ema_a       => i_ema_a, 
           i_ema_ba      => i_ema_ba, 
           i_ema_wen_dqm => i_ema_wen_dqm,  
           i_ema_cas     => i_ema_cas,  
           i_ema_ras     => i_ema_ras,  
           i_ema_sdcke   => i_ema_sdcke,  
           i_ema_rnw     => i_ema_rnw,  
                      
           -- FPGA fabric signals
           o_core_be      => be_r,  
           o_core_addr    => addr_r,  
           o_core_cs5     => arm_cs5_r,  
           o_core_cs4     => dsp_cs4_r,  
           o_core_edi     => edi_r,  
           i_core_edo5    => edo_arm,  
           i_core_edo4    => edo_dsp,  
           o_core_rd      => rd_r,  
           o_core_wr      => wr_r  
    );

------------------------------------------------------------------------------
-- Base Module
------------------------------------------------------------------------------
bm : base_module
   generic map (
      CONFIG       => "MityDSP_L138",
      GEN_DCM_RST  => TRUE,
      GEN_DNA_PORT => TRUE
      )
   port map (
      ema_clk         => ema_clk, 
      i_cs            => arm_cs5_r(CORE_BASE_MODULE),
      i_ID            => FPGA_APPLICATION_ID,
      i_version_major => FPGA_VERSION_MAJOR,
      i_version_minor => FPGA_VERSION_MINOR,
      i_year          => FPGA_YEAR,
      i_month         => FPGA_MONTH,
      i_day           => FPGA_DAY,
      i_ABus          => addr_r,
      i_DBus          => edi_r,
      o_DBus          => edo_arm(CORE_BASE_MODULE),
      i_wr_en         => wr_r,
      i_rd_en         => rd_r,
      i_be_r          => be_r,
            
      o_dcm_reset  => dcm_reset,
      i_dcm_status => dcm_status,
      i_dcm_lock   => dcm_lock,

      i_irq_map    => irq_map,
      o_irq_output => o_int                  
      
      );

tlv2548IPW : spi
   Port Map (  
      emif_clk       => ema_clk, 
      i_ABus         => addr_r,
      i_DBus         => edi_r,
      o_DBus         => edo_arm(CORE_SPI_MODULE),
      i_wr_en        => wr_r,
      i_rd_en        => rd_r,
      i_cs           => arm_cs5_r(CORE_SPI_MODULE),
      o_irq          => irq_map(CORE_SPI_IRQ_LEVEL)(CORE_SPI_IRQ_VECTOR),
      i_ilevel       => CONV_STD_LOGIC_VECTOR(CORE_SPI_IRQ_LEVEL,2),
      i_ivector      => CONV_STD_LOGIC_VECTOR(CORE_SPI_IRQ_VECTOR,4),
      
      -- SPI interface signals		
      o_sclk         => o_ads1158_sclk,
      o_cs_n         => spi_cs,
      o_mosi         => o_ads1158_din,
      i_miso         => i_ads1158_dout,
		
      -- In/Out FIFO interfaces (NO FIRST WORD FALL THROUGH)
      --    Synchronous on emif_clock
      i_fifo_depth          => "010",
      o_mosi_fifo_wr        => mosi_wr_en,
      o_mosi_fifo_rd        => mosi_rd_en,
      o_mosi_fifo_in        => mosi_din,  -- fifo data input
      i_mosi_fifo_out       => mosi_dout,
      i_mosi_write_cnt      => mosi_wr_data_count,
      i_mosi_empty     	    => mosi_empty,

      o_miso_fifo_wr        => miso_wr_en,
      o_miso_fifo_rd        => miso_rd_en,
      o_miso_fifo_in        => miso_din,
      i_miso_fifo_out       => miso_dout,
      i_miso_read_cnt       => miso_rd_data_count,
      i_miso_empty          => miso_empty,
      
      o_fifo_rst            => spi_fifo_rst

    );

-- generate 10 MHz clock
gen_clk : process(i_ema_clk)
begin
    if rising_edge(i_ema_clk) then
        if ads1158_cnt = CONV_STD_LOGIC_VECTOR(5-1, 5) then
            ads1158_clk <= not ads1158_clk;
        else
            ads1158_cnt <= ads1158_cnt+'1';
        end if;
    end if;
end process gen_clk;

o_ads1158_clkio  <= ads1158_clk;

o_ads1158_cs_n  <= spi_cs(0);

mosi_fifo : fifo_dpram64x32
		port map (
			clk => ema_clk,
			rst => spi_fifo_rst,
			din => mosi_din,
			wr_en => mosi_wr_en,
			rd_en => mosi_rd_en,
			dout => mosi_dout,
			full => open,
			empty => mosi_empty,
			data_count => mosi_wr_data_count(5 downto 0));	

miso_fifo : fifo_dpram64x32
		port map (
			clk => ema_clk,
			rst => spi_fifo_rst,
			din => miso_din,
			wr_en => miso_wr_en,
			rd_en => miso_rd_en,
			dout => miso_dout,
			full => open,
			empty => miso_empty,
			data_count => miso_rd_data_count(5 downto 0));	

gpio_inst : gpio
   Generic Map (
      NUM_BANKS       => 1,
      NUM_IO_PER_BANK => 16
   )
   Port Map ( 
      clk             => ema_clk, 
      i_ABus          => addr_r,
      i_DBus          => edi_r,
      o_DBus          => edo_arm(CORE_GPIO_MODULE),
      i_wr_en         => wr_r,
      i_rd_en         => rd_r,
      i_cs            => arm_cs5_r(CORE_GPIO_MODULE),
      o_irq           => irq_map(CORE_GPIO_IRQ_LEVEL)(CORE_GPIO_IRQ_VECTOR),
      i_ilevel        => CONV_STD_LOGIC_VECTOR(CORE_GPIO_IRQ_LEVEL,2),      
      i_ivector       => CONV_STD_LOGIC_VECTOR(CORE_GPIO_IRQ_VECTOR,4),
      i_io            => gpio_in,
      t_io            => gpio_t,
      o_io            => gpio_out,
      i_initdir       => (others=>'0'),
      i_initoutval    => (others=>'0')
   );

o_ads1158_start  <= gpio_out(0);

gpio_in(0)       <= gpio_out(0);
gpio_in(1)       <= i_ads1158_drdy;

hs_dac : dual_awg_dac5672
   Port Map ( 
      clk             => ema_clk, 
      i_ABus          => addr_r,
      i_DBus          => edi_r,
      o_DBus          => edo_arm(CORE_DAWG_MODULE),
      i_wr_en         => wr_r,
      i_rd_en         => rd_r,
      i_cs            => arm_cs5_r(CORE_DAWG_MODULE),
      o_irq           => irq_map(CORE_DAWG_IRQ_LEVEL)(CORE_DAWG_IRQ_VECTOR),
      i_ilevel        => CONV_STD_LOGIC_VECTOR(CORE_DAWG_IRQ_LEVEL,2),      
      i_ivector       => CONV_STD_LOGIC_VECTOR(CORE_DAWG_IRQ_VECTOR,4),
      
      i_cha_clock     => i_upp_cha_clock,
      i_cha_data(7 downto 0)  => i_upp_d,
      i_cha_data(15 downto 8) => i_upp_xd,
      i_cha_enable    => i_upp_cha_enable,
      i_cha_start     => i_upp_cha_start,
      o_cha_wait      => o_upp_cha_wait,

      -- DAV5672 interface
      i_dac_clk       => adc_clk, -- use 40 MHz ADC clock ema_clk, -- for now...
      o_clka          => clk1,
      o_da            => o_da,
      o_clkb          => clk2,
      o_db            => o_db
      
   );

o_clk1_ddr2 : ODDR2
		port map (
			Q  => o_clk1,
			C0 => clk1,
			C1 => not clk1,
			CE => '1',
			D0 => '1',
			D1 => '0',
			R  => '0',
			S  => '0'
		);

o_clk2_ddr2 : ODDR2
		port map (
			Q  => o_clk2,
			C0 => clk2,
			C1 => not clk2,
			CE => '1',
			D0 => '1',
			D1 => '0',
			R  => '0',
			S  => '0'
		);

HSADC : ads5560
   Port Map ( 
      clk             => ema_clk, 
      i_ABus          => addr_r,
      i_DBus          => edi_r,
      o_DBus          => edo_arm(CORE_HSADC_MODULE),
      i_wr_en         => wr_r,
      i_rd_en         => rd_r,
      i_cs            => arm_cs5_r(CORE_HSADC_MODULE),
      o_irq           => irq_map(CORE_HSADC_IRQ_LEVEL)(CORE_HSADC_IRQ_VECTOR),
      i_ilevel        => CONV_STD_LOGIC_VECTOR(CORE_HSADC_IRQ_LEVEL,2),      
      i_ivector       => CONV_STD_LOGIC_VECTOR(CORE_HSADC_IRQ_VECTOR,4),
      
      -- ADC interface
      i_adc_in_p      => i_adc_in_p,
      i_adc_in_n      => i_adc_in_n,
      i_adc_clk_p     => i_adc_clk_p,
      i_adc_clk_n     => i_adc_clk_n,
      i_adc_ovr       => i_adc_ovr,
      o_adc_clk       => open,
		
      -- UPP interface
      i_upp_clock      => upp_clock,
      o_upp_clock      => upp_clock_out,
      o_upp_d(7 downto 0)  => o_upp_d,
      o_upp_d(15 downto 8) => o_upp_xd,
      o_upp_enable     => o_upp_chb_enable,
      o_upp_start      => o_upp_chb_start,
      i_upp_wait       => i_upp_chb_wait

   );

adc_clk_ddr2 : ODDR2
		port map (
			Q  => o_upp_chb_clock,
			C0 => upp_clock_out,
			C1 => not upp_clock_out,
			CE => '1',
			D0 => '1',
			D1 => '0',
			R  => '0',
			S  => '0'
		);

------------------------------------------------------------------------------
-- Base Port Assignments
------------------------------------------------------------------------------
o_nmi_n <= '1';
io_ema_d <= ema_d when t_ema_d = '0' else (others=>'Z');
io_ema_wait_n <= ema_wait_n when t_ema_wait = '0' else (others=>'Z');

end rtl;